<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Spedition;

use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\SetFormDataEvent;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\AdminModule\Model\VatRates;
use EshopCatalog\Model\Entities\VatRate;
use EshopOrders\AdminModule\Model\ExportHelper;
use EshopOrders\AdminModule\Model\GroupsCustomers;
use EshopOrders\AdminModule\Model\Speditions;
use EshopOrders\Model\Entities\Spedition;
use EshopOrders\Model\Entities\SpeditionPriceLevel;
use EshopOrders\Model\Entities\SpeditionWeight;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Helpers\EshopOrdersCache;
use Nette\Caching\Cache;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;

class SpeditionForm extends BaseControl
{
	/** @var int @persistent */
	public $id;

	public ?Spedition          $spedition = null;
	protected Speditions       $speditionsService;
	protected ExportHelper     $exportHelper;
	protected GroupsCustomers  $groupCustomers;
	protected VatRates         $vatRates;
	protected EshopOrdersCache $eshopOrdersCache;

	public function __construct(
		Speditions       $speditions,
		ExportHelper     $exportHelper,
		GroupsCustomers  $groupCustomers,
		VatRates         $vatRates,
		EshopOrdersCache $eshopOrdersCache
	)
	{
		$this->speditionsService = $speditions;
		$this->exportHelper      = $exportHelper;
		$this->groupCustomers    = $groupCustomers;
		$this->vatRates          = $vatRates;
		$this->eshopOrdersCache  = $eshopOrdersCache;
	}

	public function render(): void
	{
		$this->template->thisForm = $this['form'];

		if (EshopOrdersConfig::load('speditions.allowWeights')) {
			$weights = [];
			if ($this->spedition) {
				foreach ($this->spedition->weights as $weight) {
					$weights[] = [
						'weightFrom' => $weight->getWeightFrom(),
						'price'      => $weight->getPrice(),
					];
				}
			}
			$this->template->weightsDefault    = $weights;
			$this->template->weightsMultiplier = [
				'name'           => 'weights',
				'fieldsetLegend' => $this->t('eshopOrders.spedition.weightPriceModifiers'),
				'fields'         => [
					'weightFrom' => [
						'label' => $this->t('eshopOrders.spedition.weightFrom'),
						'el'    => 'input',
						'attrs' => [
							'type' => 'number',
							'min'  => 0,
							'step' => 0.01,
						],
					],
					'price'      => [
						'label' => $this->t('eshopOrders.spedition.priceAdd'),
						'el'    => 'input',
						'attrs' => [
							'type' => 'number',
							'min'  => 0,
							'step' => 0.01,
						],
					],
				],
			];
		}

		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addText('name', 'default.name')
			->setRequired()
			->setMaxLength(255);
		$form->addText('ident', 'eshopOrders.paymentForm.itemIdent')
			->setDescription('eshopOrders.speditionForm.identDescription')
			->setMaxLength(255);
		$form->addText('code1', 'eshopOrders.speditionForm.code1')
			->setDescription('eshopOrders.speditionForm.code1Description')
			->setMaxLength(255)
			->setNullable();
		$form->addText('text', 'default.text')
			->setMaxLength(255);
		$form->addBool('isPublished', 'default.isActive')
			->setDefaultValue(1);
		$form->addBool('isPickup', 'eshopOrders.speditionForm.isPickup')
			->setDefaultValue(0);

		if (EshopOrdersConfig::load('orderForm.speditionsGroupByPickup')) {
			$form->addBool('isStorePickup', 'eshopOrders.speditionForm.isStorePickup')
				->setDefaultValue(0);
		}

		$form->addBool('allowOversized', 'eshopOrders.spedition.allowOversized')
			->setDefaultValue(0);
		$form->addFilesManager('image', 'default.image');
		$form->addSelect('zboziId', 'eshopOrders.speditionForm.zboziId', [null => ''] + $this->exportHelper->getZboziSpeditions());
		$form->addSelect('heurekaId', 'eshopOrders.speditionForm.heurekaId', [null => ''] + $this->exportHelper->getHeurekaSpeditions())
			->setDescription('eshopOrders.speditionForm.heurekaIdDescription');
		$form->addBool('googleId', 'eshopOrders.speditionForm.googleId')->setDefaultValue(1);
		if (isset($this->langsService->getLangs(false)['de'])) {
			$form->addSelect('idealoId', 'eshopOrders.speditionForm.idealoId', [null => ''] + $this->exportHelper->getIdealoSpeditions());
		}
		$form->addText('externalIdent', 'eshopOrders.speditionForm.externalIdent')
			->setMaxLength(255)
			->setNullable();

		$form->addSelect('vatRate', 'eshopOrders.speditionForm.vatRate', $this->vatRates->getOptionsForSelect());

		$priceText = 'eshopOrders.default.priceSimple';
		if (EshopOrdersConfig::load('speditions.priceIsWithoutVat'))
			$priceText = 'eshopOrders.default.priceSimpleWithoutVat';

		$form->addText('price', $priceText)
			->setHtmlType('number')
			->setHtmlAttribute('step', .01);
		if (EshopOrdersConfig::load('allowFreeFrom')) {
			$form->addInteger('freeFrom', 'eshopOrders.default.freeFrom');
		}
		$form->addInteger('availableFrom', 'eshopOrders.default.availableFrom')
			->setDefaultValue(0);
		$form->addInteger('availableTo', 'eshopOrders.default.availableTo')
			->setDefaultValue(999999);

		if (EshopOrdersConfig::load('allowFreeFrom')) {
			$form->addDateTimePicker('freeFromDate', 'eshopOrders.spedition.freeFromDate');
			$form->addDateTimePicker('freeToDate', 'eshopOrders.spedition.freeToDate');
		}

		if (EshopOrdersConfig::load('speditions.allowWeights')) {
			$form->addText('maxWeight', 'eshopOrders.spedition.maxWeight')
				->setHtmlType('number')
				->setHtmlAttribute('min', 0)
				->setHtmlAttribute('step', .01);
		}

		if (EshopOrdersConfig::load('enablePriceLevels', false)) {
			$container = $form->addContainer('priceLevels', 'eshopOrders.spedition.priceLevels');
			foreach ($this->groupCustomers->getOptionsForSelect() as $k => $v) {
				$group = $container->addContainer($k, $this->t('eshopOrders.spedition.priceLevel', ['group' => $v]));
				$group->addInteger('price', $priceText);
				if (EshopOrdersConfig::load('allowFreeFrom')) {
					$group->addInteger('freeFrom', 'eshopOrders.default.freeFrom');
				}
				$group->addInteger('availableFrom', 'eshopOrders.default.availableFrom');
				$group->addInteger('availableTo', 'eshopOrders.default.availableTo');

				if (EshopOrdersConfig::load('allowFreeFrom')) {
					$group->addDateTimePicker('freeFromDate', 'eshopOrders.spedition.freeFromDate');
					$group->addDateTimePicker('freeToDate', 'eshopOrders.spedition.freeToDate');
				}
			}
		}

		$this->eventDispatcher->dispatch(
			new CreateFormEvent($form, $this->getPresenter(false) ? $this->template : null),
			SpeditionForm::class . '::createForm'
		);

		$form->addSaveCancelControl('saveControl');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			if ($this->id) {
				$spedition    = $this->speditionsService->get($this->id);
				$flashMessage = 'eshopOrders.speditionForm.edited';
			} else {
				$spedition    = new Spedition();
				$flashMessage = 'eshopOrders.speditionForm.added';
			}

			$spedition->setName($values->name);
			$spedition->setIdent($values->ident);
			$spedition->setText($values->text);
			$spedition->isPublished = $values->isPublished;
			$spedition->setPrice((float) $values->price);
			$spedition->setAvailableFrom((int) $values->availableFrom);
			$spedition->setAvailableTo((int) $values->availableTo);
			$spedition->image     = $values->image;
			$spedition->zboziId   = $values->zboziId ?: null;
			$spedition->heurekaId = $values->heurekaId ?: null;
			$spedition->googleId  = $values->googleId ? '1' : '0';
			if (isset($this->langsService->getLangs(false)['de'])) {
				$spedition->idealoId = $values->idealoId ?: null;
			}
			$spedition->code1          = $values->code1 ?: null;
			$spedition->isPickup       = $values->isPickup;
			$spedition->allowOversized = $values->allowOversized ? 1 : 0;
			$spedition->vatRate        = $values->vatRate ? $this->em->getReference(VatRate::class, $values->vatRate) : null;
			$spedition->externalIdent  = $values->externalIdent;

			if (isset($values['isStorePickup'])) {
				$spedition->isStorePickup = $values->isStorePickup;
			}

			if (EshopOrdersConfig::load('speditions.allowWeights')) {
				$spedition->setMaxWeight($values->maxWeight);
			}

			if (EshopOrdersConfig::load('allowFreeFrom')) {
				$spedition->setFreeFrom(is_numeric($values->freeFrom) ? (int) $values->freeFrom : null);
			} else {
				$spedition->setFreeFrom(null);
			}

			if (EshopOrdersConfig::load('allowFreeFrom')) {
				$spedition->freeFromDate = $values->freeFromDate;
				$spedition->freeToDate   = $values->freeToDate;
			}

			if (EshopOrdersConfig::load('enablePriceLevels', false)) {
				$current = $spedition->getPriceLevels()->toArray();
				foreach ($values->priceLevels as $groupId => $vals) {
					if (!$vals->price)
						continue;

					$priceLevel = $current[$groupId] ?? null;
					if (!$priceLevel) {
						$priceLevel = new SpeditionPriceLevel($spedition, $this->groupCustomers->getReference($groupId), $vals->price);
						$spedition->getPriceLevels()->set($groupId, $priceLevel);
					}

					$priceLevel->setPrice($vals->price);
					$priceLevel->freeFrom      = EshopOrdersConfig::load('allowFreeFrom') && $vals->freeFrom ? $vals->freeFrom : null;
					$priceLevel->availableFrom = $vals->availableFrom ?: 0;
					$priceLevel->availableTo   = $vals->availableTo ?: 999999;

					if (EshopOrdersConfig::load('allowFreeFrom')) {
						$priceLevel->freeFromDate = $vals->freeFromDate;
						$priceLevel->freeToDate   = $vals->freeToDate;
					}

					unset($current[$groupId]);
					$this->em->persist($priceLevel);
				}

				foreach ($current as $priceLevel)
					$this->em->remove($priceLevel);
			}

			$httpData = $form->getHttpData();

			if (EshopOrdersConfig::load('speditions.allowWeights')) {
				$weights = [];
				foreach ($httpData['weights']['weightFrom'] ?? [] as $i => $v) {
					$price = $httpData['weights']['price'][$i] ?? 0;

					if (!$v) {
						continue;
					}

					$weights[number_format((float) $v, 2, '.', '')] = $price;
				}

				foreach ($spedition->weights as $k => $weight) {
					if (isset($weights[$k])) {
						$weight->setPrice($weights[$k]);
					} else {
						$spedition->weights->removeElement($weight);
						$this->em->remove($weight);
					}
				}

				foreach (array_diff_key($weights, $spedition->weights->toArray()) as $weightFrom => $price) {
					$weight = new SpeditionWeight($spedition, $weightFrom, $price);
					$this->em->persist($weight);
					$spedition->weights->set($weightFrom, $weight);
				}
			}

			$event                   = new FormSuccessEvent(
				$form,
				$values,
				$this->getPresenter(false) ? $this->template : null,
				$this->getPresenter(false) ? $this->getPresenter() : null);
			$event->custom['entity'] = $spedition;
			$this->eventDispatcher->dispatch($event, SpeditionForm::class . '::formSuccess');

			$this->em->persist($spedition);
			$this->em->flush();
			$form->addCustomData('speditionId', $spedition->getId());
			$this->getPresenter()->flashMessageSuccess($flashMessage);

			$this->eshopOrdersCache->getCache()->remove('speditionWeights');
			$this->eshopOrdersCache->getCache()->clean([
				Cache::Tags => ['spedition', 'freeFrom'],
			]);

			return true;
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('formErrors');
		}

		return false;
	}

	public function setSpedition(int $id): void
	{
		$this->id        = $id;
		$this->spedition = $s = $this->speditionsService->get($id);

		if (!$s)
			throw new InvalidArgumentException();

		$d = [
			'name'           => $s->getName(),
			'ident'          => $s->getIdent(),
			'code1'          => $s->code1,
			'text'           => $s->getText(),
			'isPublished'    => $s->isPublished,
			'price'          => $s->getPrice(),
			'freeFrom'       => $s->getFreeFrom(),
			'availableFrom'  => $s->getAvailableFrom(),
			'availableTo'    => $s->getAvailableTo(),
			'image'          => $s->image,
			'zboziId'        => $s->zboziId,
			'heurekaId'      => $s->heurekaId,
			'googleId'       => $s->googleId ? 1 : 0,
			'idealoId'       => $s->idealoId,
			'isPickup'       => $s->isPickup ? 1 : 0,
			'allowOversized' => $s->allowOversized ? 1 : 0,
			'vatRate'        => $s->vatRate ? $s->vatRate->getId() : null,
			'externalIdent'  => $s->externalIdent,
			'freeFromDate'   => $s->freeFromDate,
			'freeToDate'     => $s->freeToDate,
		];

		if (EshopOrdersConfig::load('orderForm.speditionsGroupByPickup')) {
			$d['isStorePickup'] = $s->isStorePickup;
		}

		if (EshopOrdersConfig::load('speditions.allowWeights')) {
			$d['maxWeight'] = str_replace(',', '.', (string) $s->getMaxWeight());
		}

		if (EshopOrdersConfig::load('enablePriceLevels', false)) {
			foreach ($s->getPriceLevels()->toArray() as $groupId => $priceLevel) {
				/** @var SpeditionPriceLevel $priceLevel */
				$d['priceLevels'][$groupId] = [
					'price'         => $priceLevel->getPrice(),
					'freeFrom'      => $priceLevel->freeFrom,
					'availableFrom' => $priceLevel->availableFrom,
					'availableTo'   => $priceLevel->availableTo,
					'freeFromDate'  => $priceLevel->freeFromDate,
					'freeToDate'    => $priceLevel->freeToDate,
				];
			}
		}

		$this['form']->setDefaults($d);

		$this->eventDispatcher->dispatch(new SetFormDataEvent($this['form'], $this->spedition), SpeditionForm::class . '::setSpedition');
	}
}
