<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Presenters;

use Contributte\Scheduler\Helpers\Debugger;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\AdminModule\Components\Invoice\IInvoiceEditFormFactory;
use EshopOrders\AdminModule\Components\Order\CustomerInfoForm;
use EshopOrders\AdminModule\Components\Order\ICorrectiveTaxDocumentFormFactory;
use EshopOrders\AdminModule\Components\Order\ICustomerInfoFormFactory;
use EshopOrders\AdminModule\Components\Order\IOrderDiscountCodeFormFactory;
use EshopOrders\AdminModule\Components\Order\IOrderDiscountFormFactory;
use EshopOrders\AdminModule\Components\Order\IOrderItemAddFormFactory;
use EshopOrders\AdminModule\Components\Order\IUploadExternalInvoiceFactory;
use EshopOrders\AdminModule\Components\Order\OrderDiscountCodeForm;
use EshopOrders\AdminModule\Model\Event\OrderNavigationCallbackEvent;
use EshopOrders\AdminModule\Model\Event\OrdersNavigationEvent;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\Model\Entities\OrderDiscount;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Orders;
use EshopOrders\Model\Subscribers\OrderSubscriber;
use Exception;
use Nette\Application\UI\InvalidLinkException;
use EshopOrders\AdminModule\Components\Order\IOrderFormFactory;
use EshopOrders\AdminModule\Components\Order\IOrdersGridFactory;
use EshopOrders\AdminModule\Components\Order\IOrderStatusFormFactory;
use EshopOrders\AdminModule\Components\Order\IOrderItemFormFactory;
use EshopOrders\AdminModule\Components\Order\IOrderSpeditionFormFactory;
use EshopOrders\AdminModule\Components\Order\IOrderPaymentFormFactory;
use EshopOrders\AdminModule\Components\Order\IOrderAddressFormFactory;
use EshopOrders\Model\Entities\Order;

class DefaultPresenter extends BasePresenter
{
	/** @var Orders @inject */
	public ?Orders $ordersService = null;

	protected function startup()
	{
		parent::startup();
		$this['title']->setTitle($this->translator->translate('eshopOrders.title.orders'));
		$this->setHeader('eshopOrders.title.orders', 'fa fa-shopping-cart');
		OrderSubscriber::$enableSendInvoice = true;
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	/** Vyrenderuje vyskakovaci formular pro pridani stavu k objednavce
	 *
	 * @param int $orderId ID objednavky
	 */
	public function handleOrderStatusForm()
	{
		$this->template->modal = 'orderStatusForm';
		$this->redrawControl('modal');

		$this->setTitle($this->t('eshopOrders.title.addOrderStatus'));
		$this->template->modalTitle = $this['title']->getTitle();
	}

	public function handleSendCustomerInfo()
	{
		$this->template->modal = 'customerInfo';
		$this->redrawControl('modal');

		$this->setTitle($this->t('eshopOrders.title.sendCustomerInfo'));
		$this->template->modalTitle = $this['title']->getTitle();
	}

	/** Vyrenderuje vyskakovaci formular pro editaci položky (cena a pocet ks)
	 *
	 * @param int $itemId ID polozky
	 */
	public function handleOrderItemForm($itemId = null)
	{
		$this->template->modal = 'orderItemForm';
		$this->redrawControl('modal');

		$control = $this['orderItemForm'];
		if ($itemId) {
			$control->setOrderItem($itemId);
			$this->setTitle($this->t('eshopOrders.title.editOrderItem'), $itemId);
		} else {
			$this->setTitle($this->t('eshopOrders.title.addOrderItem'));
		}
		$this->template->modalTitle = $this['title']->getTitle();
	}

	/** Vyrenderuje vyskakovaci formular pro pridani stavu k objednavce
	 *
	 * @param int $itemId ID objednavky
	 */
	public function handleOrderItemAddForm()
	{
		$this->template->modal = 'orderItemAddForm';
		$this->redrawControl('modal');

		$this->setTitle($this->t('eshopOrders.title.addOrderItem'), ' ');
		$this->template->modalTitle = $this['title']->getTitle();
	}

	/** Vyrenderuje vyskakovaci formular pro editaci dopravy v objednavce
	 *
	 * @param int $itemId ID objednavky
	 */
	public function handleOrderSpeditionForm()
	{
		$this->template->modal = 'orderSpeditionForm';
		$this->redrawControl('modal');

		$control = $this['orderSpeditionForm'];

		$this->setTitle($this->t('eshopOrders.title.editOrderSpedition'));

		$this->template->modalTitle = $this['title']->getTitle();
	}

	/** Vyrenderuje vyskakovaci formular pro editaci platby v objednavce
	 *
	 * @param int $itemId ID objednavky
	 */
	public function handleOrderPaymentForm()
	{
		$this->template->modal = 'orderPaymentForm';
		$this->redrawControl('modal');

		$control = $this['orderPaymentForm'];

		$this->setTitle($this->t('eshopOrders.title.editOrderPayment'));

		$this->template->modalTitle = $this['title']->getTitle();
	}

	/** Vyrenderuje vyskakovaci formular pro editaci dorucovaci adresy
	 */
	public function handleAddressDeliveryForm()
	{
		$this->template->modal = 'orderAddressDeliveryForm';
		$this->redrawControl('modal');

		$this->setTitle($this->t('eshopOrders.title.editAddressDelivery'));

		$this->template->modalTitle = $this['title']->getTitle();
	}

	/** Vyrenderuje vyskakovaci formular pro editaci dorucovaci adresy
	 */
	public function handleAddressInvoiceForm()
	{
		$this->template->modal = 'orderAddressInvoiceForm';
		$this->redrawControl('modal');

		$this->setTitle($this->t('eshopOrders.title.editAddressInvoice'));

		$this->template->modalTitle = $this['title']->getTitle();
	}

	/** Vyrenderuje vyskakovaci formular pro urceni produtku v odd
	 */
	public function handleCorrectiveTaxDocumentForm()
	{
		$order = $this->ordersService->get((int) $this->getParameter('id'));

		if ($order->getOrderForCorrectiveTaxDocument() !== null) {
			$this->redirect(':EshopOrders:Admin:Default:editOrder', $order->getOrderForCorrectiveTaxDocument()->getId());
		}

		$this->template->modal = 'correctiveTaxDocumentForm';
		$this->redrawControl('modal');

		$this->setTitle($this->t('eshopOrders.title.correctiveTaxDocumentForm'));

		$this->template->modalTitle = $this['title']->getTitle();
	}

	public function handleUploadExternalInvoice(): void
	{
		$order = $this->ordersService->get((int) $this->getParameter('id'));

		$this->template->modal = 'UploadExternalInvoice';
		$this->redrawControl('modal');

		$this->setTitle($order->getInvoice()
			? $this->t('eshopOrders.uploadExternalInvoiceForm.titleInvoiceExists', null, ['invoice' => $order->getInvoice()->ident])
			: $this->t('eshopOrders.uploadExternalInvoiceForm.title')
		);

		$this->template->modalTitle = $this['title']->getTitle();
	}

	public function handleOrderDiscountForm()
	{
		$this->template->modal = 'orderDiscountForm';
		$this->redrawControl('modal');

		$this->setTitle($this->t('eshopOrders.title.orderDiscountForm'));

		$this->template->modalTitle = $this['title']->getTitle();
	}

	public function handleOrderDiscountCodeForm(): void
	{
		$this->template->modal = 'orderDiscountCodeForm';
		$this->redrawControl('modal');

		$this->setTitle($this->t('eshopOrders.title.orderDiscountCodeForm'));

		$this->template->modalTitle = $this['title']->getTitle();
	}

	public function handleInvoiceEditForm()
	{
		$this->template->modal = 'invoiceEditForm';
		$this->redrawControl('modal');

		$this->setTitle($this->t('eshopOrders.title.editInvoice'));

		$this->template->modalTitle = $this['title']->getTitle();
	}

	public function handleNavigationCallback(string $key)
	{
		$this->eventDispatcher->dispatch(new OrderNavigationCallbackEvent($key, $this), 'eshopOrders.admin.ordersDefaultNavigationCallback');
	}

	public function handleSendActualOrderToClient(int $id)
	{
		$order = $this->ordersService->getFullOrder($id);
		if ($order)
			$this->eventDispatcher->dispatch(new OrderEvent($order), 'eshopOrders.sendActualOrder');
	}

	public function handleSendInvoice(int $id)
	{
		$order = $this->ordersService->getFullOrder($id);
		if ($order)
			$this->eventDispatcher->dispatch(new OrderEvent($order), 'eshopOrders.sendInvoice');
	}

	public function handleGridChangeStatus()
	{
		$status = $this->getParameter('status');
		$ids    = $this->getParameter('ids');

		if ($status && $ids) {
			$this['ordersGrid']->gridChangeStatus($ids, $status);
		}
	}

	public function handleRemoveOrderDiscount(int $discountId): void
	{
		try {
			$orderDiscount = $this->em->getRepository(OrderDiscount::class)->find($discountId);

			if ($orderDiscount) {
				$this->em->remove($orderDiscount);
				$this->em->flush();
			}
			$this->flashMessageSuccess('default.removed');
		} catch (Exception $exception) {
			Debugger::log($exception);
			$this->flashMessageDanger('default.error');
		}

		$this->redirect('this');
	}

	/*******************************************************************************************************************
	 * ============================== Actions
	 */

	public function actionDefault()
	{
		$navigation = [];

		foreach (EshopOrdersConfig::load('ordersGridNavButtonStatuses', []) as $status) {
			if (is_string($status)) {
				$navigation[] = [
					'title'            => $this->t('eshopOrders.ordersGrid.statusesLong.' . $status),
					'link'             => $this->link('gridChangeStatus!'),
					'data-status'      => $status,
					'data-grid-action' => $this['ordersGrid']['grid']->getFullName(),
				];
			}
		}

		$this->eventDispatcher->dispatch(new OrdersNavigationEvent($navigation), 'eshopOrders.admin.ordersDefaultNavigation');
		$this->transformNavigation($navigation);

		$this['navigation']->setData(['header' => $navigation]);
	}

	public function actionEditOrder($id)
	{
		/** @var Order $order */
		$order = $this->em->getRepository(Order::class)->find($id);

		if (!$order)
			$this->error('Order ' . $id . ' not found');

		$navigation = [
			[
				'title' => 'eshopOrders.menu.editAddressDelivery',
				'link'  => $this->link('addressDeliveryForm!'),
				'ico'   => 'plus',
				'class' => 'ajax',
			], [
				'title' => 'eshopOrders.menu.editAddressInvoice',
				'link'  => $this->link('addressInvoiceForm!', ['type' => 'invoice']),
				'ico'   => 'plus',
				'class' => 'ajax',
			], [
				'title' => 'eshopOrders.menu.addItem',
				'link'  => $this->link('orderItemAddForm!'),
				'ico'   => 'plus',
				'class' => 'ajax',
			], [
				'title' => 'eshopOrders.menu.addStatus',
				'link'  => $this->link('orderStatusForm!'),
				'ico'   => 'plus',
				'class' => 'ajax',
			], [
				'title' => 'eshopOrders.menu.sendCustomerInfo',
				'link'  => $this->link('sendCustomerInfo!'),
				'ico'   => 'envelope',
				'class' => 'ajax',
			], [
				'title' => 'eshopOrders.menu.sendActualOrder',
				'link'  => $this->link('sendActualOrderToClient!'),
				'ico'   => 'envelope',
				'class' => 'ajax',
			],
		];

		if ((EshopOrdersConfig::load('invoice.enable') && !in_array($order->getPaymentIdent(), ['storeCash',
					'storeCard'])) || $order->isCorrectiveTaxDocument) {
			if (!$order->getInvoice()) {
				if (!EshopOrdersConfig::load('invoice.allowUploadCustomInvoiceFile') || $order->enableInvoiceGeneration) {
					$navigation[] = [
						'link'   => $this->link('Orders:invoice', $order->getId()),
						'ico'    => 'fas fa-sync',
						'title'  => 'eshopOrders.menu.generateInvoice',
						'target' => '_blank',
					];
				}
			} else {
				$navigation[] = [
					'link'   => $this->link('Orders:invoice', $order->getId()),
					'ico'    => 'fas fa-file-invoice',
					'title'  => 'eshopOrders.menu.showInvoice',
					'target' => '_blank',
				];
				$navigation[] = [
					'link'  => $this->link('sendInvoice!'),
					'ico'   => 'envelope',
					'title' => 'eshopOrders.menu.sendInvoice',
					'class' => 'ajax',
				];
				if ($order->getInvoice()->getId() && !$order->getInvoice()->isExternal() && $this->getUser()->isAllowed('EshopOrders:Admin:Invoice', 'regenerateInvoices')) {
					$navigation[] = [
						'link'  => $this->link('regenerate!', $order->getInvoice()->getId()),
						'ico'   => 'sync',
						'title' => 'eshopOrders.menu.regenerateInvoice',
						'class' => 'ajax',
					];
				}
			}
			if (EshopOrdersConfig::load('invoice.allowUploadCustomInvoiceFile') && !$order->enableInvoiceGeneration) {
				$navigation[] = [
					'link'  => $this->link('uploadExternalInvoice!'),
					'ico'   => 'fas fa-file-upload',
					'title' => 'eshopOrders.menu.uploadExternalInvoice',
					'class' => 'ajax',
				];
			}
		}

		if ($order->getInvoice() && !$order->getInvoice()->isExternal()) {
			$navigation[] = [
				'link'  => $this->link('invoiceEditForm!'),
				'ico'   => 'far fa-sticky-note',
				'title' => 'eshopOrders.menu.editInvoice',
				'class' => 'ajax',
			];
		}

		if (EshopOrdersConfig::load('enableDeliveryListPrint')) {
			$defaultLang  = $this->langsService->getLangs()[$this->translator->getDefaultLocale()];
			$navigation[] = [
				'link'   => $this->link('Orders:printDeliveryList', ['ids' => [$order->getId()]]),
				'ico'    => 'file-pdf',
				'title'  => $order->lang === $defaultLang->getTag()
					? 'eshopOrders.menu.showDeliveryList'
					: $this->t('eshopOrders.menu.showDeliveryListLang', [
						'lang' => $this->langsService->getLangs(false)[$order->lang]->getShortTitle(),
					]),
				'target' => '_blank',
			];

			if ($order->lang !== $defaultLang->getTag()) {
				$navigation[] = [
					'link'   => $this->link('Orders:printDeliveryList', [
						'ids'  => [$order->getId()],
						'lang' => $defaultLang->getTag(),
					]),
					'ico'    => 'file-pdf',
					'title'  => $this->t('eshopOrders.menu.showDeliveryListLang', ['lang' => $defaultLang->getShortTitle()]),
					'target' => '_blank',
				];
			}
		}

		if (EshopOrdersConfig::load('correctiveTaxDocument.enable')) {
			if (!$order->isCorrectiveTaxDocument && $order->getOrderForCorrectiveTaxDocument() === null) {
				$navigation[] = [
					'link'  => $this->link('correctiveTaxDocumentForm!'),
					'ico'   => 'fas fa-copy',
					'title' => 'eshopOrders.menu.createCorrectiveTaxDocument',
					'class' => 'ajax',
				];
			} else if ($order->getOrderForCorrectiveTaxDocument() !== null) {
				$navigation[] = [
					'link'  => $this->link('editOrder', $order->getOrderForCorrectiveTaxDocument()->getId()),
					'ico'   => 'fas fa-copy',
					'title' => 'eshopOrders.menu.showCorrectiveTaxDocument',
				];
			}
		}

		if (EshopOrdersConfig::load('orderForm.enableOrderDiscounts')) {
			$navigation[] = [
				'link'  => $this->link('orderDiscountForm!'),
				'ico'   => 'fas fa-percentage',
				'title' => 'eshopOrders.menu.addOrderDiscount',
				'class' => 'ajax',
			];
		}

		if (class_exists('EshopSales\DI\EshopSalesExtension') && EshopOrdersConfig::load('orderForm.enableOrderDiscountCodes')) {
			$navigation[] = [
				'link'  => $this->link('orderDiscountCodeForm!'),
				'ico'   => 'fas fa-percent',
				'title' => 'eshopOrders.menu.addOrderDiscountCode',
				'class' => 'ajax',
			];
		}

		$e        = new OrdersNavigationEvent($navigation);
		$e->order = $order;
		$this->eventDispatcher->dispatch($e, 'eshopOrders.admin.onBeforeRenderEditOrderNavigation');

		$this->transformNavigation($navigation);

		$this['navigation']->setData(['header' => $navigation]);

		if (!$order->isCorrectiveTaxDocument) {
			$title = $this->t('eshopOrders.title.editOrder', ['id' => $order->getId()]);
		} else if ($order->correctiveTaxDocumentOf !== null) {
			$title = $this->t('eshopOrders.title.correctiveTaxDocumentFormWithOrder', ['order' => $order->correctiveTaxDocumentOf->getId()]);
		} else if ($order->isCorrectiveTaxDocument) {
			$title = $this->t('eshopOrders.title.correctiveTaxDocumentForm');
		}
		$this->setHeader($title, 'fa fa-shopping-cart');

		$this->template->order = $order;
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentOrdersGrid(IOrdersGridFactory $factory)
	{
		return $factory->create();
	}

	protected function createComponentOrderForm(IOrderFormFactory $factory)
	{
		$control = $factory->create();

		if ($this->getParameter('id'))
			$control->setOrder($this->getParameter('id'));

		return $control;
	}

	protected function createComponentOrderStatusForm(IOrderStatusFormFactory $factory)
	{
		$control = $factory->create();
		$control['form']->setAjax();

		if ($this->getParameter('id'))
			$control->setOrder($this->getParameter('id'));

		$control['form']->onSuccessSave[]         = function() {
			$this->redrawControl('flashes');
			$this['orderForm']->redrawStatuses();
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->payload->hideModal = true;
			$this->redrawControl('flashes');
			$this['orderForm']->redrawStatuses();
		};

		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}

	protected function createComponentOrderItemForm(IOrderItemFormFactory $factory)
	{
		$control = $factory->create();
		$control['form']->setAjax();

		if ($this->getParameter('id'))
			$control->setOrder($this->getParameter('id'));

		$control['form']->onSuccessSave[]         = function() {
			$this->redrawControl('flashes');
			$this['orderForm']->redrawControl('items');
			$this['orderForm']->redrawControl('priceTotal');
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->payload->hideModal = true;
			$this->redrawControl('flashes');
			$this['orderForm']->redrawControl('items');
			$this['orderForm']->redrawControl('priceTotal');
		};

		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}

	protected function createComponentOrderItemAddForm(IOrderItemAddFormFactory $factory)
	{
		$control = $factory->create((int) $this->getParameter('id'));

		$control->onAnchor[] = function() use ($control) {
			$control['form']->onSuccessSave[]         = function() {
				$this->redrawControl('flashes');
				$this['orderForm']->redrawControl('items');
				$this['orderForm']->redrawControl('priceTotal');
			};
			$control['form']->onSuccessSaveAndClose[] = function() {
				$this->payload->hideModal = true;
				$this->redrawControl('flashes');
				$this['orderForm']->redrawControl('items');
				$this['orderForm']->redrawControl('priceTotal');
			};

			$control['form']['saveControl']->closeModalOnCancel();
		};

		return $control;
	}

	protected function createComponentOrderSpeditionForm(IOrderSpeditionFormFactory $factory)
	{
		$control = $factory->create();
		$control['form']->setAjax();

		if ($this->getParameter('id'))
			$control->setOrder($this->getParameter('id'));

		$control['form']->onSuccessSave[]         = function() {
			$this->redrawControl('flashes');
			$this['orderForm']->redrawControl('items');
			$this['orderForm']->redrawControl('spedition');
			$this['orderForm']->redrawControl('priceTotal');
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->payload->hideModal = true;
			$this->redrawControl('flashes');
			$this['orderForm']->redrawControl('items');
			$this['orderForm']->redrawControl('spedition');
			$this['orderForm']->redrawControl('priceTotal');
		};

		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}

	protected function createComponentOrderPaymentForm(IOrderPaymentFormFactory $factory)
	{
		$control = $factory->create();
		$control['form']->setAjax();

		if ($this->getParameter('id'))
			$control->setOrder($this->getParameter('id'));

		$control['form']->onSuccessSave[]         = function() {
			$this->redrawControl('flashes');
			$this['orderForm']->redrawControl('payment');
			$this['orderForm']->redrawControl('priceTotal');
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->payload->hideModal = true;
			$this->redrawControl('flashes');
			$this['orderForm']->redrawControl('payment');
			$this['orderForm']->redrawControl('priceTotal');
		};

		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}

	protected function createComponentOrderAddressDeliveryForm(IOrderAddressFormFactory $factory)
	{
		$control = $factory->create();
		$control['form']->setAjax();

		if ($this->getParameter('id'))
			$control->setOrderAddress($this->getParameter('id'), 'delivery');

		$control['form']->onSuccessSave[]         = function() {
			$this->redrawControl('flashes');
			$this['orderForm']->redrawControl('addressDelivery');
			$this['orderForm']->redrawControl('addressInvoice');
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->payload->hideModal = true;
			$this->redrawControl('flashes');
			$this['orderForm']->redrawControl('addressDelivery');
			$this['orderForm']->redrawControl('addressInvoice');
		};

		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}

	protected function createComponentOrderAddressInvoiceForm(IOrderAddressFormFactory $factory)
	{
		$control = $factory->create();
		$control['form']->setAjax();

		if ($this->getParameter('id'))
			$control->setOrderAddress($this->getParameter('id'), 'invoice');

		$control['form']->onSuccessSave[]         = function() {
			$this->redrawControl('flashes');
			$this['orderForm']->redrawControl('addressDelivery');
			$this['orderForm']->redrawControl('addressInvoice');
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->payload->hideModal = true;
			$this->redrawControl('flashes');
			$this['orderForm']->redrawControl('addressDelivery');
			$this['orderForm']->redrawControl('addressInvoice');
		};

		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}

	protected function createComponentCorrectiveTaxDocumentForm(ICorrectiveTaxDocumentFormFactory $factory)
	{
		$control = $factory->create($this->ordersService->get((int) $this->getParameter('id')));

		$control['form']->onSuccessSave[]         = function() {
			$this->redrawControl('flashes');
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->payload->hideModal = true;
			$this->redrawControl('flashes');
		};

		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}

	protected function createComponentUploadExternalInvoice(IUploadExternalInvoiceFactory $factory)
	{
		$control = $factory->create($this->ordersService->get((int) $this->getParameter('id')));

		$control['form']->onSuccessSave[]         = function() {
			$this->redrawControl('flashes');
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->payload->hideModal = true;
			$this->redrawControl('flashes');
		};

		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}

	protected function createComponentOrderDiscountForm(IOrderDiscountFormFactory $factory)
	{
		$control = $factory->create($this->ordersService->get((int) $this->getParameter('id')));

		$control['form']->onSuccessSave[]         = function() {
			$this->redirect('this');
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->redirect('this');
		};

		return $control;
	}

	protected function createComponentOrderDiscountCodeForm(IOrderDiscountCodeFormFactory $factory): OrderDiscountCodeForm
	{
		$control = $factory->create($this->ordersService->get((int) $this->getParameter('id')));

		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			if ($form->getCustomData('isOk')) {
				$this->redirect('this');
			} else {
				$this->redrawControl('flashes');
			}
		};

		return $control;
	}

	protected function createComponentInvoiceEditForm(IInvoiceEditFormFactory $factory)
	{
		$control = $factory->create($this->ordersService->get((int) $this->getParameter('id'))->getInvoice());

		$control['form']->onSuccessSave[]         = function() {
			$this->redrawControl('flashes');
		};
		$control['form']->onSuccessSaveAndClose[] = function() {
			$this->payload->hideModal = true;
			$this->redrawControl('flashes');
		};

		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}

	protected function createComponentCustomerInfoForm(ICustomerInfoFormFactory $factory): CustomerInfoForm
	{
		return $factory->create($this->ordersService->getFullOrder((int) $this->getParameter('id')));
	}

	/*******************************************************************************************************************
	 * ============================== Others
	 */

	/**
	 * @param array $navigation
	 *
	 * @throws InvalidLinkException
	 */
	protected function transformNavigation(array &$navigation): void
	{
		foreach ($navigation as $k => $nav) {
			if (is_array($nav['link']))
				$navigation[$k]['link'] = $this->link($nav['link'][0], $nav['link'][1]);
		}
	}
}
