<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model\Subscribers;

use Core\Model\Event\ControlEvent;
use Core\Model\Event\EventDispatcher;
use Core\Model\Images\Image;
use Core\Model\UI\AbstractPresenter;
use Core\Model\UI\FrontPresenter;
use EshopOrders\FrontModule\Components\Cart\CartPreview;
use EshopOrders\Model\EshopOrdersConfig;
use Nette\Application\LinkGenerator;
use Contributte\Translation\Translator;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\FrontModule\Model\Tags;
use EshopOrders\FrontModule\Model\Carts;
use EshopOrders\FrontModule\Model\Dao;
use EshopOrders\FrontModule\Model\Event\AddedCartItemEvent;
use EshopOrders\FrontModule\Model\Event\FillDaoItemsEvent;
use EshopOrders\FrontModule\Model\Event\RemovedCartItemEvent;
use EshopOrders\FrontModule\Model\Event\UpdatedCartItemEvent;
use EshopOrders\Model\Entities\CartItem;

class CartSubscriber implements EventSubscriberInterface
{
	protected Carts           $cartsService;
	protected ProductsFacade  $productsFacade;
	protected Tags            $tagsService;
	protected Translator      $translator;
	protected LinkGenerator   $linkGenerator;
	protected EventDispatcher $eventDispatcher;

	/** @var array */
	protected $cartChanges = [];

	protected $presenter;

	public function __construct(
		Carts           $carts,
		ProductsFacade  $productsFacade,
		Tags            $tags,
		Translator      $translator,
		LinkGenerator   $linkGenerator,
		EventDispatcher $eventDispatcher
	)
	{
		$this->cartsService    = $carts;
		$this->productsFacade  = $productsFacade;
		$this->tagsService     = $tags;
		$this->translator      = $translator;
		$this->linkGenerator   = $linkGenerator;
		$this->eventDispatcher = $eventDispatcher;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			FrontPresenter::class . '::startup'      => 'frontPresenterStartup',
			FrontPresenter::class . '::beforeRender' => 'beforeRender',
			'eshopOrders.cartAddItem'                => ['onAddItem', 100],
			'eshopOrders.cartUpdateItem'             => ['onUpdateItem', 100],
			'eshopOrders.cartRemoveItem'             => ['onRemoveItem', 100],
			'eshopOrders.cartFillDaoItems'           => ['onFillDaoItems', 100],
		];
	}

	public function frontPresenterStartup(ControlEvent $event): void
	{
		/** @var AbstractPresenter $presenter */
		$presenter       = $event->control->getPresenter();
		$this->presenter = $presenter;

		$callback = function() use ($presenter): void {
			/** @var CartPreview $cartPreview */
			$cartPreview = $presenter->getComponent('cartPreview', false);
			if ($cartPreview) {
				$presenter->setPayloadAjaxPlaceholder('cartPreviewItemsCount', (string) $cartPreview->getItemsCountFormatted());
				$presenter->setPayloadAjaxPlaceholder('cartPreviewItemsCountRaw', (string) $cartPreview->getItemsCountRaw());
				$presenter->setPayloadAjaxPlaceholder('cartPreviewItemsPrice', (string) $cartPreview->getPriceFormatted());
			}

			$presenter->template->cart = $cartPreview->getCart();
			$presenter->redrawControl('cartMessage');
		};

		$this->eventDispatcher->addListener('eshopOrders.cartAddItem', $callback);
		$this->eventDispatcher->addListener('eshopOrders.cartUpdateItem', $callback);
		$this->eventDispatcher->addListener('eshopOrders.cartRemoveItem', $callback);
	}

	public function beforeRender(ControlEvent $event): void
	{
		if (!empty($this->cartChanges)) {
			$message = null;
			$event->control->redrawControl('orderCartDetail');

			if (isset($this->cartChanges['added'])) {
				$message = 'itemAdded';
			}

			if (isset($this->cartChanges['updated'])) {
				$message = 'itemUpdated';
			}

			if (isset($this->cartChanges['removed'])) {
				$message = 'itemRemoved';
			}

			if ($message) {
				$event->control->flashMessageSuccess('eshopOrdersFront.cart.' . $message);
			}

			$event->control->redrawControl('flashes');
		}
	}

	public function onAddItem(AddedCartItemEvent $event): void
	{
		$item = &$event->item;

		if ($item->productId) {
			$product = $this->productsFacade->getProduct($item->productId);
			if (!$product || (!$item->ignoreValidation && !$product->canAddToCart)) {
				return;
			}
		}

		$result = $this->cartsService->addItem($item);

		if ($result === 1) {
			$this->cartChanges['added'][] = $item;
		}
	}

	public function onUpdateItem(UpdatedCartItemEvent $event): void
	{
		$result = $this->cartsService->updateItemQuantity($event->itemId, $event->quantity, $event->moreData);

		if ($result === 1) {
			$this->cartChanges['updated'][] = $event->itemId;
		}
	}

	public function onRemoveItem(RemovedCartItemEvent $event): void
	{
		$this->cartChanges['removed'][] = $event->itemId;
		$this->cartsService->removeItem($event->itemId);
	}

	public function onFillDaoItems(FillDaoItemsEvent $event): void
	{
		$cartItems = $event->items;
		if ($this->cartsService->cDaoItems) {
			return;
		}

		$productIds                   = array_map(static fn(CartItem $ci) => $ci->getProductId(), $cartItems);

		foreach ($cartItems as $ci) {
			if ($ci->getMoreDataValue('productId')) {
				$productIds[] = (int) $ci->getMoreDataValue('productId');
			}
		}

		$productIds                   = array_filter($productIds, static fn($v) => (bool) $v);
		$products                     = $productIds ? $this->productsFacade->getProducts($productIds) : [];
		ProductsFacade::$loadFeatures = true;

		$items  = [];
		$childs = [];
		foreach ($cartItems as $ci) {
			$product = $products[$ci->getProductId()] ?? null;

			if ($product && !$product->canAddToCart && !$ci->getMoreDataValue('isCustom', false)) {
				continue;
			}

			if ($product) {
				$cartItem = new Dao\CartItem($ci->name ?: (string) $product->getName(), $product->getId(), $product->getPrice());
				$cartItem->setProduct($product)
					->setLink($product->link)
					->setVatRate($product->vatRate ?: 21);
				$cartItem->weight = $product->weight ?: 0;
			} else {
				$cartItem = new Dao\CartItem((string) $ci->name, $ci->productId, (float) $ci->price);
				$cartItem->setVatRate($ci->vatRate ?? 21);
				$cartItem->weight = $product->weight ?: 0;

				if ($ci->getMoreDataValue('productId') && isset($products[$ci->getMoreDataValue('productId')])) {
					$cartItem->setProduct($products[$ci->getMoreDataValue('productId')]);
				}
			}

			$cartItem
				->setId($ci->getId())
				->setQuantity($ci->quantity);

			$cartItem->ident = (string) $ci->getIdent();
			$cartItem->setData($ci->getMoreData());

			$cartItem->priceInBaseCurrency = ($product->priceInBaseCurrency ?? $ci->price)
				+ $ci->getMoreDataValue('extraPrice', 0);

			if ($product) {
				$cartItem->discountDisabled              = $product->discountDisabled;
				$cartItem->disableCalculateFreeSpedition = $product->disableCalculateFreeSpedition;
				$cartItem->verifyAge                     = $product->verifyAge;

				if ($product->getGallery() && $product->getGallery()->getCover()) {
					$cartItem->setImage($product->getGallery()->getCover());
				}

				if (EshopOrdersConfig::load('allowFreeFrom')) {
					$tagFree = $product->getTag('freeDelivery');

					if ($tagFree && !$tagFree->isAuto) {
						$cartItem->freeDelivery = true;
					}
				}

				if ($ci->image) {
					$cartItem->image = new Image($ci->image);
				}
			}

			if ($ci->getParent()) {
				$childs[$ci->getParent()->getId()][$ci->getId()] = $cartItem;
			} else {
				$items[$ci->getId()] = $cartItem;
			}
		}

		foreach ($childs as $parentId => $parentChilds) {
			foreach ($parentChilds as $childId => $child) {
				/** @var Dao\CartItem $child */
				if (isset($items[$parentId])) {
					$itemProduct = $items[$parentId]->product;

					if ($itemProduct && $itemProduct->package) {
						$child->priceInBaseCurrency = 0;
						$child->price               = 0;
					}

					$items[$parentId]->addChild($child);
				}
			}
		}

		$this->cartsService->cDaoItems = $items;
	}
}
