<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Presenters;

use EshopCatalog\DI\EshopCatalogExtension;
use EshopCatalog\FrontModule\Components\CartAddForm;
use EshopCatalog\FrontModule\Components\ICartAddFormFactory;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopOrders\FrontModule\Components\Cart\CartDetail;
use EshopOrders\FrontModule\Components\Order\IOrderFormFactory;
use EshopOrders\FrontModule\Components\Cart\ICartDetailFactory;
use EshopOrders\Model\Orders;
use EshopOrders\FrontModule\Model\Carts;
use EshopCatalog\Model\Config as EshopCatalogConfig;
use Nette\Application\UI\Multiplier;

/**
 * Class DefaultPresenter
 * @package EshopOrders\FrontModule\Presenters
 */
class DefaultPresenter extends BasePresenter
{
	protected Orders $ordersService;

	public Carts $cartsService;

	protected ProductsFacade $productsFacade;

	public function __construct(Orders $orders, Carts $carts, ProductsFacade $productsFacade)
	{
		$this->ordersService  = $orders;
		$this->cartsService   = $carts;
		$this->productsFacade = $productsFacade;
	}

	public function cartUpdateListener()
	{
		$this->redrawControl('orderForm');
	}

	/*******************************************************************************************************************
	 * ============================== Actions
	 */

	public function actionOrder(): void
	{
		if (class_exists(EshopCatalogExtension::class)
			&& EshopCatalogConfig::load('allowAddToCartOnlyForLoggedUsers') && !$this->getUser()->isLoggedIn())
			$this->redirectUrl($this->getHomepage()->link);
	}

	/*******************************************************************************************************************
	 * ============================== Render
	 */

	public function renderOrder(): void
	{
		$cart                        = $this->cartsService->getCurrentCart();
		$this->template->pageClass   .= ' page-cart';
		$this->template->cartIsEmpty = empty($cart->getCartItems());
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentOrderForm(IOrderFormFactory $factory)
	{
		$control = $factory->create();

		return $control;
	}

	protected function createComponentCartDetail(ICartDetailFactory $factory): CartDetail
	{
		$control = $factory->create();

		$this->eventDispatcher->addListener('eshopOrders.cartUpdateItem', [$this, 'cartUpdateListener']);
		$this->eventDispatcher->addListener('eshopOrders.cartRemoveItem', [$this, 'cartUpdateListener']);

		return $control;
	}

	protected function createComponentCartAddForm(ICartAddFormFactory $factory): Multiplier
	{
		return new Multiplier(function($productId) use ($factory): CartAddForm {
			$control = $factory->create();
			$control->setProduct($this->productsFacade->getProduct((int) $productId));
			$control['form']->onSuccess[] = function() {
				$this->redrawControl('cart');
			};

			return $control;
		});
	}
}
