<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Helpers\Strings;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Core\Model\Entities\TId;

/**
 * @ORM\Table("eshop_orders__payment")
 * @ORM\Entity
 */
class Payment
{
	use TId;

	/**
	 * @var string
	 * @ORM\Column(name="name", type="string")
	 */
	protected $name;

	/**
	 * @var string
	 * @ORM\Column(name="ident", type="string")
	 */
	protected $ident;

	/**
	 * @ORM\Column(type="string", nullable=true)
	 */
	public ?string $code1 = null;

	/**
	 * @var string
	 * @ORM\Column(name="text", type="string")
	 */
	protected $text;

	/**
	 * @var int
	 * @ORM\Column(name="is_published", type="smallint")
	 */
	public $isPublished;

	/**
	 * @var int
	 * @Gedmo\SortablePosition
	 * @ORM\Column(name="position", type="integer")
	 */
	protected $position;

	/**
	 * @var float
	 * @ORM\Column(name="price", type="decimal", precision=10, scale=2)
	 */
	protected $price;

	/**
	 * @ORM\Column(name="currency", type="string", nullable=true)
	 */
	public ?string $currency = null;

	/**
	 * @var int
	 * @ORM\Column(name="free_from", type="integer", nullable=true)
	 */
	protected $freeFrom;

	/**
	 * @var int
	 * @ORM\Column(name="available_from", type="integer", nullable=true)
	 */
	protected $availableFrom;

	/**
	 * @var int
	 * @ORM\Column(name="available_to", type="integer", nullable=true)
	 */
	protected $availableTo;

	/**
	 * @var string
	 * @ORM\Column(name="image", type="string", nullable=true, length=255)
	 */
	public $image;

	/**
	 * @var Customer[]|ArrayCollection
	 * @ORM\ManyToMany(targetEntity="Customer", mappedBy="disabledPayments", indexBy="id")
	 */
	protected $disabledForCustomers;

	/**
	 * @ORM\Column(name="disable_in_google", type="smallint", length=1, options={"default": 0})
	 */
	public int $disableInGoogle = 0;

	/**
	 * @ORM\Column(name="pohoda_name", type="string", length=255, nullable=true)
	 */
	public ?string $pohodaName = null;

	/**
	 * Member constructor.
	 */
	public function __construct()
	{
		$this->isPublished = 1;
		$this->setPosition(-1);
		$this->freeFrom             = 999999;
		$this->disabledForCustomers = new ArrayCollection();
	}

	public function getName()
	{
		return $this->name;
	}

	public function getPosition()
	{
		return $this->position;
	}

	public function getPrice()
	{
		return $this->price;
	}

	public function setName($name)
	{
		$this->name = $name;

		return $this;
	}

	public function setPosition($position)
	{
		$this->position = intval($position);

		return $this;
	}

	public function setPrice(float $price): self
	{
		$this->price = Strings::formatEntityDecimal($price);

		return $this;
	}

	/*******
	 * === Ident
	 */

	public function getIdent()
	{
		return $this->ident;
	}

	public function setIdent($ident): Payment
	{
		$this->ident = $ident;

		return $this;
	}

	/*******
	 * === Text
	 */

	public function getText()
	{
		return $this->text;
	}

	public function setText($text)
	{
		$this->text = $text;

		return $this;
	}

	/*******
	 * === AvailableFrom
	 */

	public function getAvailableFrom()
	{
		return $this->availableFrom;
	}

	public function setAvailableFrom(int $availableFrom): Payment
	{
		$this->availableFrom = $availableFrom;

		return $this;
	}

	/*******
	 * === AvailableTo
	 */

	public function getAvailableTo()
	{
		return $this->availableTo;
	}

	public function setAvailableTo(int $availableTo): Payment
	{
		$this->availableTo = $availableTo;

		return $this;
	}

	public function getFreeFrom(): ?int { return $this->freeFrom; }

	public function setFreeFrom(?int $freeFrom): self
	{
		$this->freeFrom = $freeFrom;

		return $this;
	}
}
