<?php declare(strict_types = 1);

namespace EshopOrders\Model\Helpers;

use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\Strings;
use EshopCatalog\FrontModule\Model\Dao\Seller;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\EshopOrdersConfig;
use Rikudou\CzQrPayment\QrPayment;
use rikudou\EuQrPayment\Iban\IBAN as IBANEU;
use rikudou\EuQrPayment\QrPayment as QrPaymentEU;
use rikudou\EuQrPayment\Sepa\CharacterSet;
use rikudou\EuQrPayment\Sepa\Purpose;
use Rikudou\Iban\Iban\CzechIbanAdapter;
use Rikudou\Iban\Iban\IBAN;

class QrGenerator
{
	public function createQrCode(
		string  $bankAccount,
		string  $bankCode,
		string  $variableSymbol,
		float   $amount,
		string  $currency,
		?string $iban = null,
		bool    $instantPayment = true,
		?string $sellerName = null
	): QrPayment
	{
		$qrPayment = new QrPayment(new CzechIbanAdapter($bankAccount, $bankCode));
		$qrPayment->setVariableSymbol($variableSymbol)
			->setAmount(round($amount, 2))
			->setCurrency(Strings::upper($currency))
			->setInstantPayment($instantPayment);

		if ($iban) {
			$qrPayment->setIban(new IBAN($iban));
		}

		if ($sellerName) {
			$qrPayment->setPayeeName($sellerName);
		}

		return $qrPayment;
	}

	public function createEuQrCode(
		string  $iban,
		string  $swift,
		string  $variableSymbol,
		float   $amount,
		string  $currency,
		?string $sellerName = null
	): QrPaymentEU
	{
		$qrPayment = new QrPaymentEU(new IBANEU($iban));
		$qrPayment->setCharacterSet(CharacterSet::UTF_8)
			->setSwift($swift)
			->setBeneficiaryName($sellerName)
			->setAmount($amount)
			->setCurrency($currency)
			->setPurpose(Purpose::EPAYMENT)
			->setRemittanceText($variableSymbol);

		return $qrPayment;
	}

	public function getQrCodeByOrderAndSeller(Order $order, Seller $seller, ?string $variableSymbol = null)
	{
		$bankAccount = $seller->getBankAccount($order->getCurrencyCode());

		if (!$bankAccount || !$bankAccount->numberPart1 || !$bankAccount->numberPart2) {
			return null;
		}

		$sepaCountrySkip = ['cz'];

		if (!EshopOrdersConfig::load('orderForm.sepaQrCodeSkipSk')) {
			$sepaCountrySkip[] = 'sk';
		}

		if (
			EshopOrdersConfig::load('orderForm.allowSepaQrCode')
			&& $order->getAddressInvoice()
			&& $order->getAddressInvoice()->getCountry()
			&& !Arrays::contains($sepaCountrySkip, Strings::lower($order->getAddressInvoice()->getCountry()->getId()))) {

			return $this->createEuQrCode(
				$bankAccount->iban,
				$bankAccount->swift,
				$variableSymbol ?: ((string) $order->getId()),
				$order->getPrice(true),
				$order->getCurrencyCode(),
				$seller->name ? Strings::truncate($seller->name, 35, '') : null,
			);
		}

		return $this->createQrCode(
			(string) $bankAccount->numberPart1,
			(string) $bankAccount->numberPart2,
			$variableSymbol ?: ((string) $order->getId()),
			$order->getPrice(true),
			$order->getCurrencyCode(),
			$bankAccount->iban,
			true,
			$seller->name ? Strings::truncate($seller->name, 35, '') : null,
		);
	}
}
