<?php declare(strict_types = 1);

namespace EshopOrders\Model;

use Core\Model\Helpers\BaseEntityService;
use Doctrine\DBAL\LockMode;
use Doctrine\ORM\NonUniqueResultException;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\QueryBuilder;
use EshopCatalog\Model\Entities\Seller;
use EshopOrders\Model\Entities\Invoice;
use EshopOrders\Model\Entities\InvoiceConfig;
use EshopOrders\Model\Entities\NumericalSeries;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\SellerInvoiceConfig;
use Exception;
use Nette\InvalidArgumentException;
use Nette\Utils\DateTime;
use Nette\Utils\Strings;

class InvoiceConfigRepository extends BaseEntityService
{
	/** @var string */
	protected $entityClass = InvoiceConfig::class;

	/**
	 * @return QueryBuilder
	 */
	public function getQueryBuilder(): QueryBuilder
	{
		return $this->getEr()->createQueryBuilder('ic');
	}

	/**
	 * @param int $sellerId
	 *
	 * @return DateTime|null
	 * @throws Exception
	 */
	public function getDueDate(int $sellerId): ?DateTime
	{
		$now           = new DateTime;
		$defaultConfig = $this->getConfigBySeller($sellerId);

		if (!$defaultConfig) {
			return null;
		}

		return $now->modify(sprintf('+ %s days', $defaultConfig->maturity));
	}

	/**
	 * @param int $sellerId
	 *
	 * @return string|null
	 * @throws NonUniqueResultException
	 */
	public function generateIdent(int $sellerId, string $type = InvoiceConfig::TYPE_INVOICE): ?string
	{
		$defaultConfig = $this->getConfigBySeller($sellerId);
		if (!$defaultConfig) {
			return null;
		}

		switch ($type) {
			case InvoiceConfig::TYPE_INVOICE:
				$nsId = $defaultConfig->numericalSeries->getId();
				break;
			case InvoiceConfig::TYPE_CORRECTIVE:
				$nsId = $defaultConfig->correctiveNumericalSeries->getId();
				break;
			case InvoiceConfig::TYPE_RECEIPT:
				$nsId = $defaultConfig->receiptNumericalSeries->getId();
				break;
			default:
				throw new InvalidArgumentException('Value in $type arg is unknown');
		}

		/** @var NumericalSeries $numericalSeries */
		$numericalSeries = $this->em->getRepository(NumericalSeries::class)->createQueryBuilder('ns')
			->where('ns.id = :id')
			->setParameter('id', $nsId)
			->setMaxResults(1)
			->getQuery()
			->setLockMode(LockMode::PESSIMISTIC_WRITE)
			->getOneOrNullResult();

		$containsPrefixYearWildcards  = $numericalSeries->containsPrefixYearWildcards();
		$containsPrefixMonthWildcards = $numericalSeries->containsPrefixMonthWildcards();
		$startNumber                  = $numericalSeries->startNumber;
		$getLastDocumentData          = function() use ($type) {
			switch ($type) {
				case InvoiceConfig::TYPE_INVOICE:
				case InvoiceConfig::TYPE_CORRECTIVE:
					$qb = $this->em->getRepository(Invoice::class)->createQueryBuilder('i');
					$qb->join('i.order', 'o', Join::WITH, 'o.invoice = i.id')
						->andWhere('o.isCorrectiveTaxDocument = :isCorrectiveTaxDocument')
						->orderBy('i.createdAt', 'desc')
						->setParameter('isCorrectiveTaxDocument', (int) ($type === InvoiceConfig::TYPE_CORRECTIVE))
						->setMaxResults(1);

					/** @var Invoice|null $lastInvoice */
					$lastInvoice = $qb->getQuery()->getOneOrNullResult();
					if (!$lastInvoice) {
						return null;
					}

					return ['ident' => $lastInvoice->ident, 'createdAt' => $lastInvoice->createdAt];
				case InvoiceConfig::TYPE_RECEIPT:
					$qb = $this->em->getRepository(Order::class)->createQueryBuilder('o');
					$qb->where('o.receiptIdent IS NOT NULL')
						->orderBy('o.receiptIdent', 'desc')
						->setMaxResults(1);

					/** @var Order|null $lastReceiptOrder */
					$lastReceiptOrder = $qb->getQuery()->getOneOrNullResult();

					if (!$lastReceiptOrder) {
						return null;
					}

					return ['ident' => $lastReceiptOrder->receiptIdent, 'createdAt' => $lastReceiptOrder->getCreatedTime()];
				default:
					throw new InvalidArgumentException('Value in $type arg is unknown');
			}
		};

		if ($containsPrefixMonthWildcards || $containsPrefixYearWildcards) {
			$lastDocumentData = $getLastDocumentData();

			if ($lastDocumentData !== null) {
				$created = $lastDocumentData['createdAt'];
				$now     = new DateTime;

				// prefix contain month/year wildcards and current month/year is bigger then on the invoice
				if (($containsPrefixMonthWildcards && (((int) $now->format('n')) > ((int) $created->format('n')))) ||
					($containsPrefixYearWildcards && (((int) $now->format('Y')) > ((int) $created->format('Y'))))) {
					$this->resetInvoiceCounter($sellerId, $type);
					$startNumber = 1;
				}
			}
		}

		$strMaxOrder = Strings::padRight('9', $numericalSeries->digitsCount, '9');

		if (((int) $strMaxOrder) === $startNumber) {
			$lastDocumentData = $getLastDocumentData();

			if (Strings::endsWith($lastDocumentData['ident'], $strMaxOrder)) {
				return null;
			}
		}

		$prefix = $numericalSeries->getRealPrefix();
		$order  = Strings::padLeft((string) $startNumber, $numericalSeries->digitsCount, '0');

		return sprintf('%s%s', $prefix, $order);
	}

	/**
	 * @param int    $sellerId
	 * @param string $type
	 *
	 * @throws NonUniqueResultException
	 */
	private function resetInvoiceCounter(int $sellerId, string $type = InvoiceConfig::TYPE_INVOICE): void
	{
		$defaultConfig = $this->getConfigBySeller($sellerId);

		if (!$defaultConfig) {
			return;
		}

		switch ($type) {
			case InvoiceConfig::TYPE_INVOICE:
				$numericalSeries = $defaultConfig->numericalSeries;
				break;
			case InvoiceConfig::TYPE_CORRECTIVE:
				$numericalSeries = $defaultConfig->correctiveNumericalSeries;
				break;
			case InvoiceConfig::TYPE_RECEIPT:
				$numericalSeries = $defaultConfig->receiptNumericalSeries;
				break;
			default:
				throw new InvalidArgumentException('Value in $type arg is unknown');
		}

		$numericalSeries->startNumber = 1;

		$this->em->persist($numericalSeries);
		$this->em->flush();
	}

	/**
	 * @return SellerInvoiceConfig[]
	 */
	public function getUsed(): array
	{
		$qb = $this->em->createQueryBuilder();
		$qb->select('sic')
			->from(SellerInvoiceConfig::class, 'sic')
			->join('sic.seller', 's');

		return $qb->getQuery()->getResult();
	}

	/**
	 * @return bool
	 */
	public function haveAllSellersInvoiceSettingsCreated(): bool
	{
		return $this->em->getRepository(Seller::class)->count([]) === count($this->getUsed());
	}

	/**
	 * @param int $sellerId
	 *
	 * @return InvoiceConfig|null
	 * @throws NonUniqueResultException
	 */
	public function getConfigBySeller(int $sellerId): ?InvoiceConfig
	{
		$qb = $this->em->createQueryBuilder();
		$qb->select('ic')
			->from(InvoiceConfig::class, 'ic')
			->join('ic.sellerInvoiceConfigs', 'sic')
			->join('sic.seller', 's')
			->where('s.id = :id')
			->setParameter('id', $sellerId)
			->setMaxResults(1);

		return $qb->getQuery()->getOneOrNullResult();
	}

}
