<?php declare(strict_types=1);


namespace EshopOrders\Model;

use Contributte\Scheduler\Helpers\Debugger;
use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\BaseEntityService;
use EshopOrders\AdminModule\Components\Order\OrderItemAddForm;
use EshopOrders\AdminModule\Components\Order\OrderItemForm;
use EshopOrders\AdminModule\Model\Event\OrderItemEvent;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderItem;
use Exception;

/**
 * class OrderItems
 * @package EshopOrders\Model
 *
 * @method OrderItem|object|null getReference($id)
 * @method OrderItem[]|null getAll()
 * @method OrderItem|null get($id)
 */
class OrderItems extends BaseEntityService
{
	protected $entityClass = OrderItem::class;
	protected EventDispatcher $eventDispatcher;

	public function __construct(EventDispatcher $eventDispatcher)
	{
		$this->eventDispatcher = $eventDispatcher;
	}

	public function setReadyToDelivery(int $id, int $value): bool
	{
		try {
			$oi = $this->get($id);

			if (!$oi) {
				return false;
			}

			$oi->isReadyToDelivery = $value;
			$this->em->persist($oi);
			$this->em->flush($oi);

			return true;
		} catch (Exception $e) {
			Debugger::log($e);
			return false;
		}
	}

	public function addOrderItem(OrderItem $orderItem): void
	{
		$this->em->persist($orderItem);

		$this->eventDispatcher->dispatch(new OrderItemEvent($orderItem), OrderItemAddForm::class . '::formSuccess');

		$this->em->flush();
	}

	public function removeOrderItem(int $orderItemId): void
	{
		$orderItem = $this->get($orderItemId);
		$this->eventDispatcher->dispatch(new OrderItemEvent($orderItem), OrderItemForm::class . '::beforeDelete');

		$this->em->remove($orderItem);
		$this->em->flush();
	}

	public function saveOrderItem(OrderItem $orderItem): void
	{
		$this->eventDispatcher->dispatch(new OrderItemEvent($orderItem), OrderItemForm::class . '::beforeSave');

		$this->em->persist($orderItem)->flush();
	}
	
}
