<?php declare(strict_types = 1);

namespace EshopOrders\Model\Pdf;

use Core\Model\Helpers\Arrays;
use Core\Model\Http\PdfResponse;
use Core\Model\Pdf\Pdf;
use Core\Model\Sites;
use Currency\Model\Currencies;
use EshopCatalog\FrontModule\Model\Sellers;
use EshopOrders\Model\Entities\Invoice;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Helpers\QrGenerator;
use EshopOrders\Model\InvoiceConfigRepository;
use EshopOrders\Model\Invoices;
use Mpdf\Mpdf;
use Mpdf\MpdfException;
use Mpdf\Output\Destination;
use Contributte\Translation\Translator;
use Nette\Utils\FileSystem;
use Nette\Utils\Strings;

class InvoicePdfBuilder
{
	protected Invoice                 $invoice;
	protected InvoiceConfigRepository $invoiceConfigRepository;
	protected Invoices                $invoices;
	protected Translator              $translator;
	protected Pdf                     $pdfService;
	protected Currencies              $currencies;
	protected Sites                   $sites;
	protected QrGenerator             $qrGenerator;
	protected Sellers                 $sellers;

	public function __construct(
		Invoice                 $invoice,
		InvoiceConfigRepository $invoiceConfigRepository,
		Invoices                $invoices,
		Translator              $translator,
		Pdf                     $pdfService,
		Currencies              $currencies,
		Sites                   $sites,
		QrGenerator             $qrGenerator,
		Sellers                 $sellers
	)
	{
		$this->invoice                 = $invoice;
		$this->invoiceConfigRepository = $invoiceConfigRepository;
		$this->invoices                = $invoices;
		$this->translator              = $translator;
		$this->pdfService              = $pdfService;
		$this->currencies              = $currencies;
		$this->sites                   = $sites;
		$this->qrGenerator             = $qrGenerator;
		$this->sellers                 = $sellers;
	}

	/**
	 * @param string $destination
	 *
	 * @return string
	 * @throws MpdfException
	 */
	public function render(string $destination = Destination::INLINE): ?string
	{
		if ($this->invoice->isExternal()) {

			$file        = WWW_DIR . $this->invoice->getExternalInvoiceFile();
			$fileContent = FileSystem::read($file);
			if ($destination === Destination::STRING_RETURN) {
				return $fileContent;
			}

			PdfResponse::sendResponse(basename($file), $fileContent, $destination);
		}

		$pdf = $this->build();

		return $pdf->Output($this->getFileName(), $destination);
	}

	/**
	 * @return Mpdf
	 * @throws MpdfException
	 */
	private function build(): Mpdf
	{
		$order = $this->invoice->order;

		// css
		$customCssFile  = TEMPLATES_DIR . '/Front/default/EshopOrders/_pdf/invoiceStyle.css';
		$defaultCssFile = SRC_DIR . '/eshoporders/src/Templates/_pdf/invoiceStyle.css';

		// latte
		$customTemplateFile  = TEMPLATES_DIR . '/Front/default/EshopOrders/_pdf/invoice.latte';
		$defaultTemplateFile = SRC_DIR . '/eshoporders/src/Templates/_pdf/invoice.latte';

		// build template
		$this->translator->setLocale($order->lang);
		$template = $this->pdfService->getTemplateFactory();
		$template->setFile(file_exists($customTemplateFile) ? $customTemplateFile : $defaultTemplateFile);
		$template->originTemplate = $defaultTemplateFile;
		$template->lang           = $order->lang;

		$template->invoice = $this->invoice;

		$order->renderPriceDecimals                 = 2;
		\Core\Model\Helpers\Strings::$priceKeepTail = true;

		$seller = $this->sellers->getSellerForSite($order->site->getIdent());

		$customer = $order->getCustomer();
		$variableSymbol = (string) $order->getId();
		if (
			$customer
			&& $customer->getGroupCustomers()
			&& Arrays::contains(EshopOrdersConfig::load('invoice.qrPaymentUseInvoiceVsForCustomerGroup'), $customer->getGroupCustomers()->getId())
		) {
			$variableSymbol = $this->invoice->ident;
		}


		$template->order    = $order;
		$template->supplier = $this->invoice->invoiceData->getSupplier();
		$template->bank     = $this->invoice->invoiceData->getSupplier()->getBank();
		$template->qrCode   = EshopOrdersConfig::load('invoice.showQrPayment') && $seller && $order->getPaymentIdent() === 'transfer'
			? $this->qrGenerator->getQrCodeByOrderAndSeller($order, $seller, $variableSymbol)
			: null;

		$invoiceData = $this->invoice->invoiceData;

		$invoiceData->spedition->invoiceData = &$invoiceData;
		$invoiceData->payment->invoiceData   = &$invoiceData;

		$template->subscriber      = $invoiceData->getCustomer();
		$template->invoiceData     = $invoiceData;
		$template->defaultCurrency = $this->currencies->getDefaultCode();

		$this->translator->setLocale($invoiceData->lang);

		// service config
		$pdf = $this->pdfService->newPdf();
		$pdf->WriteHTML(file_get_contents(file_exists($customCssFile) ? $customCssFile : $defaultCssFile), 1);
		$pdf->WriteHTML($template->renderToString(), 0);
		$pdf->SetTitle($this->getTitle());

		$site = $this->sites->getSites()[$this->invoice->order->site->getIdent()];

		if ($site) {
			$site->currentLang = $invoiceData->lang;

			$pdf->SetHTMLFooter('<table style="border-top: 1px solid black"><tr><td style="padding-top: 1mm">' .
				(EshopOrdersConfig::load('invoice.pdfFooterText') ?: 'E-shop ' . ($site->getCurrentDomain()->siteName ?: $site->getSiteName()) . ' - ' . $site->getCurrentDomain()->getDomain()) .
				'</td></tr></table>');
		}

		return $pdf;
	}

	/**
	 * @return string
	 */
	private function getTitle(): string
	{
		return $this->translator->translate('eshopOrders.title.invoice', null, ['id' => $this->invoice->ident]);
	}

	/**
	 * @return string
	 */
	public function getFileName(): string
	{
		return sprintf('%s.pdf', Strings::webalize($this->getTitle()));
	}

}
