<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model\Dao;

use Kdyby\Doctrine\Entities\MagicAccessors;

class Cart
{
	use MagicAccessors;

	/** @var int */
	public $id;

	/** @var string */
	public $ident;

	/** @var CartItem[] */
	public $cartItems;

	/** @var Spedition */
	protected $spedition;

	/** @var Payment */
	protected $payment;

	/** @var Discount[] */
	protected $discounts = [];

	public function setId(int $id)
	{
		$this->id = $id;

		return $this;
	}

	public function setIdent($ident)
	{
		$this->ident = $ident;

		return $this;
	}

	public function setCartItems($cartItems)
	{
		$this->cartItems = $cartItems;

		return $this;
	}

	public function getId(): int
	{
		return $this->id;
	}

	public function getIdent()
	{
		return $this->ident;
	}

	/** @return CartItem[] */
	public function getCartItems(): array
	{
		return $this->cartItems;
	}

	public function getCartItemsPrice(): float
	{
		$priceTotal = 0;
		foreach ($this->cartItems as $item) {
			$priceTotal += $item->getTotalPrice();
		}

		return $priceTotal;
	}

	public function getCartItemPriceWithoutVat(): float
	{
		$priceTotal = 0;
		foreach ($this->cartItems as $item) {
			$priceTotal += $item->getTotalPriceWithoutVat();
		}

		return $priceTotal;
	}

	public function getPriceTotal(): float
	{
		$priceTotal = $this->calculateDiscounts($this->getCartItemsPrice());

		if ($this->spedition)
			$priceTotal += $this->spedition->getPriceActual($this);
		if ($this->payment)
			$priceTotal += $this->payment->getPriceActual($this);

		return $priceTotal;
	}

	public function getPriceTotalWithoutVat(): float
	{
		$priceTotal = $this->calculateDiscounts($this->getCartItemPriceWithoutVat());

		if ($this->spedition)
			$priceTotal += $this->spedition->getPriceActual($this);
		if ($this->payment)
			$priceTotal += $this->payment->getPriceActual($this);

		return $priceTotal;
	}

	public function getPriceTotalWithoutSpedition(): float
	{
		return $this->calculateDiscounts($this->getCartItemsPrice());
	}

	public function getItemsCount(): int
	{
		$count = 0;
		foreach ($this->cartItems as $item) {
			$count += $item->getQuantity();
		}

		return $count;
	}

	public function addDiscount(string $key, Discount $discount): self
	{
		$this->discounts[$key] = $discount;
		$discount->calculateDiscount($this->getCartItemsPrice());

		return $this;
	}

	public function removeDiscount(string $key): self
	{
		unset($this->discounts[$key]);

		return $this;
	}

	public function getDiscount(string $key): ?Discount { return $this->discounts[$key] ?? null; }

	/** @return Discount[] */
	public function getDiscounts(): iterable { return $this->discounts ?: []; }

	public function setSpedition(Spedition $spedition): self
	{
		$spedition->setCart($this);
		$this->spedition = $spedition;

		return $this;
	}

	public function setPayment(Payment $payment): self
	{
		$payment->setCart($this);
		$this->payment = $payment;

		return $this;
	}

	protected function calculateDiscounts(float $price): float
	{
		foreach ($this->discounts as $discount) {
			$price += $discount->calculateDiscount($price);
		}

		return $price;
	}
}


