<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use EshopOrders\Model\Entities\Payment;
use EshopOrders\FrontModule\Model\Dao;

/**
 * TODO netahat celé entity z db
 *
 * Class Payments
 * @package EshopOrders\FrontModule\Model
 */
class Payments extends BaseFrontEntityService
{
	/** @var string */
	protected $entityClass = Payment::class;

	/**
	 * @param int $id
	 *
	 * @return Dao\Payment|null
	 * @throws \Doctrine\ORM\NonUniqueResultException
	 */
	public function get($id)
	{
		$paymentRaw = $this->loadRawPayment($id);
		if (!$paymentRaw) {
			return null;
		}
		$payment = $this->fillDao([$paymentRaw])[$id];

		return $payment;
	}

	/** Pozor - vraci dopravy nezavisle na hodnote kosiku, takze i ty ktere nemusi byt platne
	 * @return Dao\Payment[] vsechny aktivni dopravy
	 */
	public function getAllPublished()
	{
		$paymentsRawQuery = $this->getEr()->createQueryBuilder('p', 'p.id');
		$paymentsRawQuery->andWhere('p.isPublished = :isPublished')->setParameter('isPublished', true);
		$paymentsRawQuery->orderBy('p.position');
		$paymentsRaw = $paymentsRawQuery->getQuery()->getResult();

		$payments = $this->fillDao($paymentsRaw);

		return $payments;
	}

	/**
	 * @param int $cartValue cena objednavky
	 *
	 * @return Dao\Payment[] vsechny dopravy, ktere jsou aktivni, a jsou dostupne pro danou cenu objednavky
	 */
	public function getAllByCartValue($cartValue)
	{
		$paymentsRawQuery = $this->getEr()->createQueryBuilder('p', 'p.id');
		$paymentsRawQuery->andWhere('p.isPublished = :isPublished')->setParameter('isPublished', true);
		$paymentsRawQuery->andWhere('p.availableFrom IS NULL OR p.availableFrom <= :cartValue')->setParameter('cartValue', $cartValue);
		$paymentsRawQuery->andWhere('p.availableTo IS NULL OR p.availableTo >= :cartValue');
		$paymentsRawQuery->orderBy('p.position');
		$paymentsRaw = $paymentsRawQuery->getQuery()->getResult();

		$payments = $this->fillDao($paymentsRaw);

		return $payments;
	}

	public function getFirstFreePayment()
	{
		$paymentQuery = $this->getEr()->createQueryBuilder('p', 'p.id')->andWhere('p.freeFrom > 0')->setMaxResults(1)->orderBy('p.freeFrom', 'ASC');
		$paymentRaw   = $paymentQuery->getQuery()->getOneOrNullResult();

		$firstFreePayment = $paymentRaw ? current($this->fillDao([$paymentRaw])) : null;

		return $firstFreePayment;
	}

	public function getRawReference($id)
	{
		return $this->getEr()->getReference($id);
	}

	/**
	 * @param int $id
	 *
	 * @return Payment
	 */
	private function loadRawPayment($id)
	{
		$paymentRawQuery = $this->getEr()->createQueryBuilder('p', 'p.id');
		$paymentRawQuery->andWhere('p.id = :id')->setParameter('id', $id);

		$paymentRaw = $paymentRawQuery->getQuery()->getOneOrNullResult();

		return $paymentRaw;
	}

	/**
	 * @param Payment[] $paymentsRaw
	 *
	 * @return Dao\Payment[]
	 */
	protected function fillDao($paymentsRaw)
	{
		$payments = [];
		foreach ($paymentsRaw as $v) {
			$payment = new Dao\Payment();
			$payment
				->setId($v->getId())
				->setIdent($v->getIdent())
				->setName($v->getName())
				->setText($v->getText())
				->setPublished($v->isPublished)
				->setPosition($v->getPosition())
				->setPrice((float) $v->getPrice())
				->setFreeFrom($v->getFreeFrom())
				->setAvailableTo($v->getAvailableTo())
				->setImage($v->image);
			$payments[$payment->getId()] = $payment;
		}

		return $payments;
	}

}

