import localforage from "localforage";
import {FormatMoney} from 'core/js/utils/formatMoney';

export default class Cart {
	constructor(wrap) {
		const self = this;
		this.wrap  = wrap;

		this.cartId           = wrap.dataset.cartId;
		this.firstLoaded      = false;
		this.baseProductPrice = parseInt(wrap.dataset.baseProductPrice);
		this.baseProductWrap  = self.wrap.querySelector('.my-package__box-price');

		this.baseProductId           = document.eshopPackageBaseProduct.id;
		this.baseProductName         = document.eshopPackageBaseProduct.name;
		this.baseProductManufacturer = document.eshopPackageBaseProduct.manufacturer;
		this.baseProductCategory     = document.eshopPackageBaseProduct.category;

		this.productsListWrap  = self.wrap.querySelector('.my-package__products-list[data-section="1"]');
		this.productsListWrap2 = self.wrap.querySelector('.my-package__products-list[data-section="2"]');
		this.itemHolder        = self.wrap.querySelector('.my-package__product.hidden');
		this.currency          = self.wrap.dataset.currency;
		this.currentFilled     = 0;
		this.maxProducts       = self.wrap.dataset.maxProducts;
		this.type              = self.wrap.dataset.type;
		this.valueMin          = self.wrap.dataset.valueMin;
		this.valueMax          = self.wrap.dataset.valueMax;
		this.totalPrice        = 0;

		this.cart = localforage.createInstance({
			name: 'eshopPackagesCart_' + this.cartId,
		});

		this.boxes = self.wrap.querySelectorAll('.products__box[data-group="boxes"]');

		if (this.boxes.length > 0) {
			this.boxes.forEach(box => {
				const submitBtn = box.querySelector('.my-package__boxes-button');
				if (submitBtn) {
					submitBtn.addEventListener('click', e => {
						const activeSubmitBtn = self.wrap.querySelector('.my-package__boxes-button.active');
						if (activeSubmitBtn) {
							activeSubmitBtn.classList.remove('active');
							activeSubmitBtn.innerText = document.eshopPackageTexts.choose;
						}

						submitBtn.classList.add('active');
						submitBtn.innerText = document.eshopPackageTexts.selected;

						const boxImageEl = box.querySelector('img');
						const boxNameEl  = box.querySelector('.cd__name');
						const boxPriceEl = box.querySelector('.product-preview__price');

						if (boxImageEl) {
							self.baseProductWrap.querySelector('img').setAttribute('src', boxImageEl.getAttribute('src'));
						}

						if (boxNameEl) {
							self.baseProductWrap.querySelector('.my-package__product--text-name').innerHTML = boxNameEl.innerHTML;
						}

						if (boxPriceEl) {
							self.baseProductWrap.querySelector('.my-package__product--text-price').innerHTML = boxPriceEl.innerText;

							self.baseProductPrice        = parseInt(boxPriceEl.dataset.price);
							self.baseProductId           = box.dataset.itemId;
							self.baseProductName         = box.dataset.itemName;
							self.baseProductManufacturer = box.dataset.itemBrand;
							self.baseProductCategory     = box.dataset.itemCategory;

							this.cart.setItem('baseProduct', self.baseProductId);
							console.log('click', this.cart.getItem('baseProduct'));
						}

						self.renderTotalPrice();
					});
				}
			});
		}

		wrap.addEventListener('click', async e => {
			let target = e.target;

			if (target.matches('.cart-add__submit')) {
				let parent = target.parentElement;
				let input  = parent.querySelector('input');

				if (!input) {
					parent = parent.parentElement;
					input  = parent.querySelector('input');
				}

				if (input) {
					let canAddToCart = false;

					const inputValue = parseInt(input.value);
					const inputMax   = parseInt(input.dataset.max);

					if (inputValue < 1)
						input.value = 1;
					else if (inputValue > inputMax)
						input.value = inputMax;

					const wrap = target.closest('.products__box');
					if (wrap.dataset.group == 2 && parent.parentElement.querySelector('[name="addNote[' + input.dataset.itemId + ']"]')) {
						const checkedEl = parent.parentElement.querySelectorAll('[name="addNote[' + input.dataset.itemId + ']"]:checked');
						const textEl    = parent.parentElement.querySelector('textarea');

						wrap.querySelectorAll('.has-error').forEach(el => {
							el.classList.remove('has-error');
						});

						if (checkedEl.length === 0) {
							wrap.querySelectorAll('.choose-text label').forEach(e => {
								e.classList.add('has-error');
							});
						} else if (checkedEl[0].value === 'yes' && textEl.value == '') {
							textEl.classList.add('has-error');
						} else {
							canAddToCart = true;
						}
					} else {
						canAddToCart = true;
					}

					if (canAddToCart) {
						await this.addToCart(input.dataset.itemId, input.value, wrap);
					}
				}
			} else if (target.matches('.cart-add__quantity--close')) {
				const wrap = target.closest('.my-package__product');

				if (wrap) {
					await this.removeItem(wrap.dataset.itemId, wrap);
				}
			} else if (target.matches('.minus')) {
				let input = target.nextElementSibling;
				let val   = input.value;

				if (val > 0) {
					val--;


					input.dataset.oldValue = input.value;
					input.value            = val;

					if (target.dataset.updateItem == 'true') {
						await self.updateQuantity(input.dataset.itemId, val);
					}
				} else {
					e.preventDefault();
					return false;
				}
			} else if (target.matches('.plus')) {
				let input = target.previousElementSibling;
				let val   = input.value;

				if (val >= parseInt(input.dataset.max))
					val = input.dataset.max;
				else
					val++;

				input.dataset.oldValue = input.value;
				input.value            = val;

				if (target.dataset.updateItem == 'true')
					await self.updateQuantity(input.dataset.itemId, val);
			} else if (target.matches('.add-to-cart-btn')) {
				await self.sendToCart();
			} else if (target.matches('.my-package__content--arrow')) {
				target.closest('.my-package').classList.toggle('content-active');
				e.preventDefault();
				return false;
			}
		});

		wrap.addEventListener('focusin', e => {
			let target = e.target;

			if (target.matches('.quantity-input')) {
				target.dataset.oldValue = target.value;
			}
		});

		wrap.addEventListener('focusout', async e => {
			let target = e.target;

			if (target.matches('.quantity-input')) {
				await self.updateQuantity(target.dataset.itemId, target.value);
			}
		});

		wrap.addEventListener('change', e => {
			let target = e.target;

			if (target.matches('.add-note-check')) {
				let prodWrap = target.closest('.products__box--right');
				let checked  = document.querySelector('[name="' + target.getAttribute('name') + '"]:checked');

				if (checked && prodWrap) {
					let textarea = prodWrap.querySelector('textarea');
					if (checked.value == 'yes')
						textarea.classList.remove('hidden');
					else
						textarea.classList.add('hidden');

				}
			}
		});

		this.cart.getItem('baseProduct').then(id => {
			if (id === null) {
				this.boxes[0].querySelector('.my-package__boxes-button').dispatchEvent(new Event('click'));
			}

			let selected = false;
			this.boxes.forEach(box => {
				if (box.dataset.itemId == id) {
					box.querySelector('.my-package__boxes-button').dispatchEvent(new Event('click'));

					selected = true;
					return true;
				}
			})

			if (!selected) {
				this.boxes[0].querySelector('.my-package__boxes-button').dispatchEvent(new Event('click'));
			}
		});

		this.getCartItems().then(e => {
			for (let i in e) {
				// e[i].data = ItemHelper.parseData(this, e[i].itemId);
			}

			self.renderItems(e);
			self.renderTotalPrice();

			try {
				let content = document.querySelector('.my-package__content');
			} catch (e) {
			}
		});
	}

	renderTotalPrice() {
		let self    = this;
		let wrap    = this.wrap.querySelector('.my-package__total-price--price');
		let valueEl = this.wrap.querySelector('.my-package__fill [data-current-value]');

		if (!wrap)
			return;

		self.getCartItems().then(e => {
			let totalPrice     = self.baseProductPrice;
			let totalValue     = 0;
			let totalValueLeft = 0;

			for (let i in e) {
				let item      = e[i];
				let itemValue = item.data.value * item.quantity;

				totalPrice += ItemHelper.getTotalPrice(item);

				let valPlaceholder = document.querySelector('.my-package__product[data-item-id="' + item.itemId + '"] [data-holder-value]');
				if (valPlaceholder) {
					valPlaceholder.textContent = itemValue;
				}

				totalValue += itemValue;
			}

			self.totalPrice  = totalPrice;
			wrap.textContent = FormatMoney.format(totalPrice, 0) + ' ' + self.currency;

			if (valueEl) {
				valueEl.textContent = totalValue;

				totalValueLeft = self.valueMax - totalValue;
			}

			let addToCartBtn = this.wrap.querySelector('.add-to-cart-btn');
			if (addToCartBtn) {
				if (totalValue >= self.valueMin && totalValue <= self.valueMax) {
					addToCartBtn.classList.add('active');
				} else {
					addToCartBtn.classList.remove('active');
				}
			}

			this.wrap.querySelectorAll('.products__box').forEach(el => {
				let itemVal = parseInt(el.dataset.value);

				if (itemVal > totalValueLeft) {
					el.classList.add('disabled');
				} else {
					el.classList.remove('disabled');
				}
			});
		});
	}

	renderItems(items) {
		const self = this;

		for (let i in items) {
			let item = items[i];

			self.renderItem(item);
		}

		self.renderTotalPrice();
	}

	renderItem(item) {
		let el = this.wrap.querySelector('.my-package__product[data-item-id="' + item.itemId + '"]');

		if (!el) {
			el = this.itemHolder.cloneNode(true);

			if (item.data.section == 2) {
				this.productsListWrap2.appendChild(el);
			} else {
				this.productsListWrap.appendChild(el);
			}
		}

		el.classList.remove('hidden');
		el.dataset.section                                 = item.data.section;
		el.querySelector('[data-holder-name]').textContent = item.data.name;
		el.querySelector('[data-holder-img]').src          = item.data.image;

		let quantityHolder = el.querySelector('.quantity-input');
		if (quantityHolder) {
			quantityHolder.value          = item.quantity;
			quantityHolder.dataset.max    = item.data.maxQuantity;
			quantityHolder.dataset.itemId = item.itemId;
		}

		let priceHolder = el.querySelector('[data-holder-price]');
		if (priceHolder) {
			priceHolder.innerHTML = FormatMoney.format(ItemHelper.getTotalPrice(item), 0) + ' ' + this.currency;

			if (item.quantity > 1) {
				priceHolder.innerHTML += '<span>' + item.quantity + 'x</span>';
			}
		}

		el.dataset.itemId = item.itemId;

		let productBox = document.querySelector('.products__box[data-item-id="' + item.itemId + '"]');
		if (productBox) {
			let input = productBox.querySelector('input');
			if (input) {
				input.value = item.quantity;
			}

			let btn = productBox.querySelector('.cart-add__submit');
			if (btn) {
				btn.textContent = btn.dataset.inCartText;
				productBox.classList.add('in-cart');
				btn.classList.add('active');
			}
		}
	}

	async updateQuantity(itemId, quantity) {
		const self = this;
		let items  = await this.getCartItems();
		let item   = items[itemId] || null;

		if (!item) {
			let prod = this.wrap.querySelector('.products__box[data-item-id="' + itemId + '"]');
			if (prod) {
				let addBtn = prod.querySelector('.cart-add__submit');

				if (addBtn) {
					addBtn.click();
				}
			}
			return true;
		}

		if (quantity <= 0) {
			await this.removeItem(itemId);
			return;
		}

		item.quantity = quantity;
		items[itemId] = item;

		if (self.validateQuantity(items)) {
			await this.cart.setItem('items', items).then(e => {
				this.renderItem(item);
				self.renderTotalPrice();
			});
		} else {
			document.querySelectorAll('.my-package__product .quantity-input').forEach(input => {
				if (input.dataset.oldValue !== undefined)
					input.value = input.dataset.oldValue;
			});
		}
	}

	async removeItem(itemId, wrap) {
		const self = this;
		let items  = await this.getCartItems();

		let productBox = document.querySelector('.products__box[data-item-id="' + itemId + '"]');
		if (productBox) {
			let input = productBox.querySelector('input');
			if (input) {
				input.value = 0;
			}

			let btn = productBox.querySelector('.cart-add__submit');
			if (btn) {
				btn.textContent = btn.dataset.addCartText;
				productBox.classList.remove('in-cart');
				btn.classList.remove('active');
			}
		}

		let rightProd = document.querySelector('.my-package__product[data-item-id="' + itemId + '"]');
		if (rightProd) {
			rightProd.remove();
		}

		delete items[itemId];

		await this.cart.setItem('items', items).then(e => {
			if (wrap) {
				wrap.remove();
			}

			self.renderTotalPrice();
		});
	}

	clearCart() {
		this.cart.setItem('baseProduct', {});
		this.cart.setItem('items', []);
	}

	async addToCart(itemId, quantity, prodWrap = null) {
		const self = this;
		let items  = await this.getCartItems();
		let item   = items[itemId] || null;

		if (item) {
			item.quantity = parseInt(quantity);

			if (prodWrap && prodWrap.classList.contains('disabled') === false) {
				item.quantity++;
			}

			item.data = ItemHelper.parseData(self, itemId);
		} else {
			item = new CartItem(self, itemId, quantity);
		}

		item.note = ItemHelper.parseNote(self, itemId);

		if (item.quantity > item.data.maxQuantity)
			item.quantity = item.data.maxQuantity;

		items[itemId] = item;

		if (self.validateQuantity(items)) {
			this.cart.setItem('items', items).then(e => {
				self.renderItems(e);
			});
		}
	}

	async getCartItems() {
		const self = this;
		return await this.cart.getItem('items').then(e => {
			if (e) {
				const preloaded = document.eshopPackagePreloadedItems;
				if (preloaded && !self.firstLoaded) {
					let items = {};
					for (let i in preloaded) {
						let rawItem = preloaded[i];
						let item    = {
							itemId  : rawItem.id,
							quantity: rawItem.quantity,
						}

						if (rawItem.noteCheck) {
							item.note = {
								noteCheck: rawItem.noteCheck,
								note     : rawItem.note,
							}
						}

						item.data = ItemHelper.parseData(self, rawItem.id);

						items[rawItem.id] = item;
					}
					self.cart.setItem('items', items);
					self.firstLoaded = true;

					return items;
				}

				if (!self.firstLoaded) {
					for (let i in e) {
						let item  = e[i];
						item.data = ItemHelper.parseData(self, e[i].itemId);

						let listEl = self.wrap.querySelector('.products__box[data-item-id="' + item.itemId + '"]');
						if (listEl && item.note) {
							let noteCheckEl = listEl.querySelector('input[value="' + item.note.noteCheck + '"]');

							if (noteCheckEl) {
								if (item.note.noteCheck)
									listEl.querySelector('input[value="' + item.note.noteCheck + '"]').checked = true;
								if (item.note.note)
									listEl.querySelector('textarea').value = item.note.note;
							}
						}

						e[i] = item;
					}

					self.cart.setItem('items', e);
				}
				self.firstLoaded = true;

				return e;
			}
			return [];
		});
	}

	validateQuantity(items) {
		let self = this;

		if (!self.wrap.dataset.maxProducts)
			return true;

		let totalQuantity = 0;

		for (let i in items) {
			let item = items[i];

			if (!item || item.data.section == '2')
				continue;

			totalQuantity = parseInt(item.quantity);
		}

		if (totalQuantity > self.maxProducts) {
			return false;
		} else {
			return true;
		}
	}

	async sendToCart() {
		const self = this;

		try {
			if (gtag !== undefined) {
				gtag("event", "add_to_cart", {
					currency: document.querySelector('[property="product:price:currency"]').getAttribute('content'),
					value   : self.totalPrice,
					items   : [
						{
							item_id       : self.baseProductId,
							item_name     : self.baseProductName,
							item_brand    : self.baseProductManufacturer,
							item_category3: self.baseProductCategory,
							price         : self.totalPrice,
							quantity      : 1
						}
					]
				});
			}
		} catch (e) {
		}

		let data = await self.getCartItems();

		try {
			data = data.filter(function (el) { return el; })
		} catch (e) {
		}

		global.naja.makeRequest('post', self.wrap.dataset.submitLink, {
			baseProduct: self.baseProductId,
			items      : data,
		}, {
			history: false,
		}).then(resp => {
			if (resp.cartLink) {
				self.clearCart()
				window.location.href = resp.cartLink
			}
		});
	}
}

class CartItem {
	constructor(cart, itemId, quantity = 0) {
		this.itemId   = itemId;
		this.quantity = parseInt(quantity);

		this.data = ItemHelper.parseData(cart, itemId);
	}
}

class ItemHelper {
	static parseData(cart, itemId) {
		const el = document.querySelector('.products__box[data-item-id="' + itemId + '"]');
		let data = {};

		if (!el)
			return data;

		let quantityEl = el.querySelector('input');

		data = {
			name       : el.dataset.itemName ? el.dataset.itemName : el.querySelector('.product-preview__name').textContent.trim(),
			image      : el.querySelector('.products__box--img').src,
			price      : parseFloat(el.querySelector('.product-preview__price').dataset.price.replace(',', '.')),
			maxQuantity: quantityEl.dataset.max,
			value      : parseInt(el.dataset.value),
			section    : el.dataset.itemSection,
		};

		return data
	}

	static parseNote(cart, itemId) {
		const el = document.querySelector('.products__box[data-item-id="' + itemId + '"]');

		if (el) {
			const noteEl = el.querySelector('textarea');
			let data     = {};

			if (noteEl) {
				let noteCheck  = el.querySelector('[name="addNote[' + el.dataset.itemId + ']"]:checked');
				data.noteCheck = noteCheck ? noteCheck.value : null;
				data.note      = noteEl.value;
			}

			return data;
		}

		return {};
	}

	static getTotalPrice(item) {
		return item.quantity * item.data.price;
	}
}
