<?php declare(strict_types = 1);

namespace EshopPackages\AdminModule\Components\Packages;

use Core\Model\UI\AdminPresenter;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Core\Model\UI\Form\Controls\TextInput;
use EshopCatalog\AdminModule\Model\Categories;
use EshopCatalog\AdminModule\Model\Features;
use EshopCatalog\AdminModule\Model\FeaturesFacade;
use EshopCatalog\AdminModule\Model\Manufacturers;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\Model\Entities\Product;
use EshopPackages\AdminModule\Model\Services\EditablePackagesService;
use EshopPackages\Model\Entities\EditablePackage;
use EshopPackages\Model\Entities\EditablePackageBox;
use EshopPackages\Model\Entities\EditablePackageText;
use EshopPackages\Model\Helper;
use Exception;
use Nette\Application\LinkGenerator;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;

class EditablePackageForm extends BaseControl
{
	/** @var int|null @persistent */
	public ?int                       $id = null;
	protected EditablePackagesService $editablePackages;
	protected FeaturesFacade          $featuresFacade;
	protected Features                $features;
	protected Manufacturers           $manufacturers;
	protected Categories              $categories;
	protected Products                $products;
	protected Helper                  $helper;
	protected LinkGenerator           $linkGenerator;

	public function __construct(
		EditablePackagesService $editablePackages,
		FeaturesFacade          $featuresFacade,
		Features                $features,
		Manufacturers           $manufacturers,
		Categories              $categories,
		Products                $products,
		Helper                  $helper,
		LinkGenerator           $linkGenerator
	)
	{
		$this->editablePackages = $editablePackages;
		$this->featuresFacade   = $featuresFacade;
		$this->features         = $features;
		$this->manufacturers    = $manufacturers;
		$this->categories       = $categories;
		$this->products         = $products;
		$this->helper           = $helper;
		$this->linkGenerator    = $linkGenerator;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$labelPrefix = 'eshopPackages.editablePackage.';

		$form->addText('title', $labelPrefix . 'title')
			->setIsMultilanguage();
		$form->addText('shortTitle', $labelPrefix . 'shortTitle')
			->setIsMultilanguage();
		$form->addText('videoLink', $labelPrefix . 'videoLink')
			->setIsMultilanguage();
		$form->addEditor('text', $labelPrefix . 'text')
			->setIsMultilanguage();
		$form->addFilesManager('icon', $labelPrefix . 'icon');
		$form->addText('btnText', $labelPrefix . 'btnText')
			->setIsMultilanguage();
		$form->addCheckboxNestedList('categories', $labelPrefix . 'categories', $this->categories->getFlatTree());
		$form->addCheckboxNestedList('features', $labelPrefix . 'features', $this->featuresFacade->getFlatTree());
		$form->addCheckboxList('manufacturers', $labelPrefix . 'manufacturers', $this->manufacturers->getOptionsForSelect());
		$form->addCheckboxNestedList('categories2', $labelPrefix . 'categories', $this->categories->getFlatTree());
		$form->addCheckboxNestedList('features2', $labelPrefix . 'features', $this->featuresFacade->getFlatTree());
		$form->addCheckboxList('manufacturers2', $labelPrefix . 'manufacturers', $this->manufacturers->getOptionsForSelect());

		$form->addSelect('limitType', $labelPrefix . 'limitType', [
			'products'     => $labelPrefix . 'limitTypeList.products',
			'featureValue' => $labelPrefix . 'limitTypeList.featureValue',
		])->setDefaultValue('products');
		$form->addText('maxProducts', $labelPrefix . 'maxProducts');
		$form->addSelect('valueFeature', $labelPrefix . 'valueFeature', ['' => ''] + $this->features->getOptionsForSelect());
		$form->addText('maxValueLimit', $labelPrefix . 'maxValueLimit');
		$form->addText('minValueLimit', $labelPrefix . 'minValueLimit');

		$boxesContainer = $form->addContainer('boxesContainer');
		$boxesContainer->addText('product', 'eshopPackages.editablePackage.searchProduct')
			->setHtmlAttribute('data-package-container-name', 'boxesContainer')
			->setHtmlAttribute('data-autocomplete-name', 'packageBoxProduct')
			->setHtmlAttribute('data-autocomplete-keys', 'id,code1,ean,name')
			->setOmitted();

		$form->monitor(AdminPresenter::class, function(AdminPresenter $presenter) use ($boxesContainer) {
			/** @var ?TextInput $productComponent */
			$productComponent = $boxesContainer->getComponent('product', false);
			if ($productComponent) {
				$productComponent->setHtmlAttribute('data-autocomplete-url', $this->linkGenerator->link('EshopCatalog:Cron:Products:loadAll'));
			}
		});

		$form->addSaveCancelControl();

		$form->onValidate[] = [$this, 'formValidate'];
		$form->onSuccess[]  = [$this, 'formSuccess'];

		return $form;
	}

	public function formValidate(BaseForm $form): void
	{
		$boxes = [];
		foreach ($form->getHttpData()['boxesContainer'] as $k => $v) {
			if (is_numeric($v)) {
				$boxes[] = (int) $v;
			}
		}

		if ($boxes === []) {
			$form->addError('eshopPackages.editablePackage.boxMustNotBeEmpty');
		}

		if ($form->hasErrors()) {
			$this->redrawControl('form');
		}
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		$this->em->beginTransaction();
		try {
			$langValues = $form->convertMultilangValuesToArray();
			/** @var EditablePackageText[] $texts */
			$texts = [];

			if ($this->id) {
				/** @var EditablePackage $editablePackage */
				$editablePackage = $this->editablePackages->get($this->id);
				$texts           = $editablePackage->getTexts()->toArray();
			} else {
				$editablePackage = new EditablePackage();
			}

			foreach ($langValues as $lang => $v) {
				if (!isset($texts[$lang]) && $v['title']) {
					$texts[$lang] = new EditablePackageText($editablePackage, $lang, $v['title']);
				}

				if (!$v['title']) {
					if (isset($texts[$lang])) {
						$this->em->refresh($texts[$lang]);
					}
					unset($texts[$lang]);
					continue;
				}

				foreach (['title', 'shortTitle', 'text', 'videoLink', 'btnText'] as $c) {
					$texts[$lang]->{$c} = $v[$c];
				}

				$this->em->persist($texts[$lang]);
			}

			$editablePackage->setTexts($texts);
			$editablePackage->icon = $values->icon;
			$editablePackage->setCategories($values->categories);
			$editablePackage->setManufacturers($values->manufacturers);
			$editablePackage->setFeatures($values->features);
			$editablePackage->setCategories2($values->categories2);
			$editablePackage->setManufacturers2($values->manufacturers2);
			$editablePackage->setFeatures2($values->features2);

			$editablePackage->limitType     = $values->limitType;
			$editablePackage->maxProducts   = $values->maxProducts ? (int) $values->maxProducts : null;
			$editablePackage->valueFeature  = $values->valueFeature ? (int) $values->valueFeature : null;
			$editablePackage->maxValueLimit = $values->maxValueLimit ? (int) $values->maxValueLimit : null;
			$editablePackage->minValueLimit = $values->minValueLimit ? (int) $values->minValueLimit : null;

			$this->em->persist($editablePackage);
			$this->em->flush();
			$this->em->commit();

			// Boxy
			$boxes = [];
			foreach ($form->getHttpData()['boxesContainer'] as $k => $v) {
				if (is_numeric($v)) {
					$boxes[] = (int) $v;
				}
			}

			$currentBoxes = $editablePackage->boxes->getKeys();

			// Delete
			foreach (array_diff($currentBoxes, $boxes) as $productId) {
				$this->em->remove($editablePackage->boxes[$productId]);
			}

			// Add
			foreach (array_diff($boxes, $currentBoxes) as $productId) {
				/** @var ?Product $product */
				$product = $this->em->getRepository(Product::class)->find($productId);

				if ($product) {
					$editablePackage->boxes[$productId] = new EditablePackageBox($editablePackage, $product);

					$this->em->persist($editablePackage->boxes[$productId]);
				}
			}

			// Set defaults
			foreach ($editablePackage->boxes as $box) {
				$product = $box->getProduct();

				$product->disableStackInCart = 1;

				$this->em->persist($product);
			}
			$this->em->flush();

			$conn = $this->em->getConnection();
			foreach ($boxes as $position => $productId) {
				$conn->executeStatement("UPDATE eshop_package__editable_package_box SET `position` = :position WHERE `product_id` = :productId AND `package_id` = :packageId", [
					'position' => $position,
					'productId' => $productId,
					'packageId' => $editablePackage->getId()
				]);
			}

			$form->addCustomData('editablePackageId', $editablePackage->getId());
			$this->presenter->flashMessageSuccess('default.saved');

			return true;
		} catch (Exception $e) {
			if ($this->em->getConnection()->isTransactionActive()) {
				$this->em->rollback();
			}
			$form->addError($e->getMessage());
			$this->redrawControl('form');
		}

		return false;
	}

	public function setEditablePackage(int $id): void
	{
		$this->id = $id;
		$ep       = $this->editablePackages->get($id);

		if (!$ep) {
			throw new InvalidArgumentException("Editable package '{$id}' not found");
		}

		$d = [
			'icon'           => $ep->icon,
			'categories'     => $ep->getCategories(),
			'manufacturers'  => $ep->getManufacturers(),
			'features'       => $ep->getFeatures(),
			'categories2'    => $ep->getCategories2(),
			'manufacturers2' => $ep->getManufacturers2(),
			'features2'      => $ep->getFeatures2(),
			'maxProducts'    => $ep->maxProducts,
			'limitType'      => $ep->limitType,
			'valueFeature'   => $ep->valueFeature,
			'maxValueLimit'  => $ep->maxValueLimit,
			'minValueLimit'  => $ep->minValueLimit,
		];

		/** @var EditablePackageText[] $texts */
		$texts = $ep->getTexts()->toArray();

		foreach ($texts as $lang => $vals) {
			foreach (['title', 'shortTitle', 'text', 'videoLink', 'btnText'] as $v) {
				$d[$v][$lang] = $vals->{$v};
			}
		}

		$this['form']->setDefaults($d);

		$boxesContainer = $this['form']->getComponent('boxesContainer', false);
		if ($boxesContainer) {
			$boxes = [];
			foreach ($ep->boxes as $box) {
				$product = $box->getProduct();

				$boxes[$product->getId()] = [
					'code1' => $product->code1,
					'name'  => $product->getText()->name,
				];
			}

			$boxesContainer->addCustomData('defaults', $boxes);
		}
	}
}
