<?php declare(strict_types = 1);

namespace EshopPackages\AdminModule\Components\Packages;

use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Doctrine\ORM\Query\Expr\Join;
use EshopPackages\AdminModule\Model\Services\EditablePackagesService;
use EshopPackages\Model\Entities\EditablePackage;

class EditablePackagesGrid extends BaseControl
{
	protected EditablePackagesService     $packagesService;
	protected IEditablePackageFormFactory $formFactory;
	protected IProductsSortFactory        $productsSortFactory;

	public function __construct(
		EditablePackagesService     $packagesService,
		IEditablePackageFormFactory $formFactory,
		IProductsSortFactory        $productsSortFactory
	)
	{
		$this->packagesService     = $packagesService;
		$this->formFactory         = $formFactory;
		$this->productsSortFactory = $productsSortFactory;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ===========================  HANDLE
	 */

	public function handleAdd(): void
	{
		$this->template->modalTitle       = $this->t('eshopPackages.title.addEditablePackage');
		$this->template->modal            = 'form';
		$this->template->modalDialogClass = 'modal-xxl';
		$this->redrawControl('modal');
	}

	public function handleEdit(int $id): void
	{
		$this['form']->id                 = $id;
		$this->template->modalTitle       = $this->t('eshopPackages.title.editEditablePackage');
		$this->template->modal            = 'form';
		$this->template->modalDialogClass = 'modal-xxl';
		$this->redrawControl('modal');
	}

	public function handleDelete(int $id): void
	{
		$presenter = $this->presenter;

		if ($this->packagesService->remove($id)) {
			$presenter->flashMessageSuccess('default.removed');
		} else {
			$presenter->flashMessageDanger('default.removeFailed');
		}

		$presenter->redrawControl('flashes');
		$this['grid']->reload();
	}

	public function handleShowSort(int $id): void
	{
		$this['productsSort']->packageId = $id;
		$this->template->modalTitle      = $this->t('eshopPackages.title.productsSort', ['package' => $this['productsSort']->getEditablePackage()->getText()->title]);
		$this->template->modal           = 'productsSort';
		$this->redrawControl('modal');
	}

	/*******************************************************************************************************************
	 * ===========================  COMPONENTS
	 */

	public function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$qb = $this->packagesService->getEr()->createQueryBuilder('p')
			->addSelect('pt')
			->innerJoin('p.texts', 'pt', Join::WITH, 'pt.lang = :lang')
			->setParameters([
				'lang' => $this->translator->getLocale(),
			]);

		$grid->setDataSource($qb);

		// Columns
		$grid->addColumnText('title', 'eshopPackages.editablePackage.title')->setRenderer(function(EditablePackage $row) {
			return $row->getText($this->translator->getLocale())->title;
		});

		// Actions
		$grid->addAction('productsSort', '', 'showSort!')->setIcon('sort')->addClass('ajax');
		$grid->addAction('edit', '', 'edit!')->setIcon('fas fa-pencil-alt')->setBsType('primary')
			->addClass('ajax');
		$grid->addAction('delete', '', 'delete!')->setIcon('fas fa-times')->setBsType('danger')
			->addCLass('ajax')->setConfirm('default.reallyDelete');

		return $grid;
	}

	protected function createComponentForm(): EditablePackageForm
	{
		$control = $this->formFactory->create();

		if ($this->getParameter('id')) {
			$control->setEditablePackage((int) $this->getParameter('id'));
		}

		$control->onAnchor[] = function() use ($control) {
			$control['form']->onSuccessSave[] = function() {
				$this->getPresenter()->redrawControl('flashes');
				$this['grid']->reload();
			};

			$control['form']->onSuccessSaveAndClose[] = function() {
				$this->getPresenter()->payload->hideModal = true;
				$this->getPresenter()->redrawControl('flashes');
				$this['grid']->reload();
			};

			$control['form']->getComponent('saveControl')->closeModalOnCancel();
		};

		return $control;
	}

	protected function createComponentProductsSort(): ProductsSort
	{
		$control            = $this->productsSortFactory->create();
		$control->packageId = (int) $this->getParameter('id');

		$control->onAnchor[] = function() use ($control) {
			$control['form']->onSuccessSave[] = function() {
				$this->getPresenter()->redrawControl('flashes');
			};

			$control['form']->onSuccessSaveAndClose[] = function() {
				$this->getPresenter()->payload->hideModal = true;
				$this->getPresenter()->redrawControl('flashes');
			};

			$control['form']->getComponent('saveControl')->closeModalOnCancel();
		};

		return $control;
	}
}
