<?php declare(strict_types = 1);

namespace EshopPackages\FrontModule\Presenters;

use Currency\Model\Currencies;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\Model\Helpers\ProductConditionEnum;
use EshopOrders\FrontModule\Model\CartFacade;
use EshopPackages\Model\Helper;
use EshopPackages\Model\Navigation\EditablePackage;
use EshopPackages\Model\Services\EditablePackagesService;
use EshopPackages\Model\Services\PackageProducts;
use Nette\Utils\Strings;
use Pages\FrontModule\Model\TemplatePages;
use Pages\Model\Helpers\TemplatePagesHelper;
use Pages\Model\Paths;

class DefaultPresenter extends BasePresenter
{
	protected TemplatePages           $templatePagesService;
	protected EditablePackagesService $editablePackages;
	protected PackageProducts         $packageProducts;
	protected Paths                   $pathsService;
	protected TemplatePagesHelper     $pagesHelper;
	protected CartFacade              $cartFacade;
	protected Helper                  $helper;
	protected Currencies              $currencies;
	protected ProductsFacade          $productsFacade;

	public ?\EshopPackages\Model\Entities\EditablePackage $editablePackage = null;
	public ?Product                                       $baseProduct;

	public function __construct(
		TemplatePages           $templatePages,
		EditablePackagesService $editablePackages,
		Paths                   $paths,
		TemplatePagesHelper     $templatePagesHelper,
		PackageProducts         $packageProducts,
		CartFacade              $cartFacade,
		Helper                  $helper,
		Currencies              $currencies,
		ProductsFacade          $productsFacade
	)
	{
		parent::__construct();
		$this->templatePagesService = $templatePages;
		$this->editablePackages     = $editablePackages;
		$this->pathsService         = $paths;
		$this->pagesHelper          = $templatePagesHelper;
		$this->packageProducts      = $packageProducts;
		$this->cartFacade           = $cartFacade;
		$this->helper               = $helper;
		$this->currencies           = $currencies;
		$this->productsFacade       = $productsFacade;
	}

	/*******************************************************************************************************************
	 * ============================== Handle
	 */

	public function handleShowModal(int $prodId): void
	{
		$product = $this->packageProducts->getProductsFacade()->getProduct($prodId);

		if ($product) {
			$this->template->modalProduct = $product;

			$file    = $this->template->getFile();
			$fileLen = Strings::length($file);
			$file    = Strings::substring($file, 0, $fileLen - 6);

			$this->setLayout(false);
			echo $this->template->renderToString($file . '_modal.latte');
			exit;
		}
		$this->sendJson([]);
	}

	/*******************************************************************************************************************
	 * ============================== Actions
	 */

	public function actionEditablePackage(int $id): void
	{
		$nav     = $this->getActiveNavigation();
		$package = $this->editablePackages->get((int) $nav->componentParams['package']);

		if (!$package) {
			$this->error('pages.page.notFound');
		}

		$cartItem = $this->getParameter('cartItem');
		if ($cartItem) {
			$item = $this->cartFacade->getCart()->getCartItems()[$cartItem] ?? null;

			if ($item) {
				$data = [];
				foreach ($item->getChilds() as $child) {
					$note   = $child->getData('note');
					$data[] = [
						'id'        => $child->productId,
						'quantity'  => $child->getQuantity(),
						'noteCheck' => $note ? 'yes' : null,
						'note'      => $note ?: null,
					];
				}

				$this->template->preloadedItems = $data;
			}
		}

		$groups  = $this->packageProducts->getProductsForEditablePackage($package);
		$groups2 = $this->packageProducts->getProductsForEditablePackage($package, true);

		$categories  = [];
		$categories2 = [];

		$forRemove = [];
		foreach ($groups2 as $group => $products) {
			foreach ($products as $prodId => $product) {
				/** @var Product $product */
				if (!isset($categories2[$product->defaultCategoryId])) {
					$categories2[$product->defaultCategory->name] = $product->defaultCategory;
				}

				$forRemove[] = $prodId;
			}
		}

		foreach ($groups as $group => $products) {
			foreach ($products as $prodId => $product) {
				/** @var Product $product */
				if (!isset($categories[$product->defaultCategoryId])) {
					$categories[$product->defaultCategory->name] = $product->defaultCategory;
				}

				if (in_array($prodId, $forRemove)) {
					unset($groups[$group][$prodId]);
				}
			}
		}

		$this->template->pageClass      .= ' eshopPackages-editablePackage ep-' . $package->getId();
		$this->template->package        = $package;
		$this->template->texts          = $package->getText();
		$this->template->groups         = $groups;
		$this->template->groups2        = $groups2;
		$this->template->categories     = $categories;
		$this->template->categories2    = $categories2;
		$this->template->productsFacade = $this->productsFacade;
		$this->template->submitLink     = $this->link(':EshopPackages:Cron:Default:addToCartEditablePackage', [
			'id'               => $package->getId(),
			'cartItem'         => $cartItem,
			'activeNavigation' => null,
		]);

		$boxesProductsIds = [];
		$firstBox         = null;
		$productId        = null;

		foreach ($package->boxes as $box) {
			if ($firstBox === null) {
				$firstBox  = $box;
				$productId = $box->getProduct()->getId();
			}

			$boxesProductsIds[] = $box->getProduct()->getId();
		}

		$boxes                 = $boxesProductsIds !== [] ? $this->productsFacade->getProducts($boxesProductsIds) : [];
		$this->template->boxes = $boxes;

		$productDao        = $productId && isset($boxes[$productId]) ? $boxes[$productId] : null;
		$this->baseProduct = $productDao;

		if ($productDao) {
			$this->template->baseProduct = $productDao;
			$this->editablePackage       = $package;

			$url = $this->link('//this');

			$currency = $this->currencies->getCurrent();
			if (!$currency) {
				$this->error('eshopCatalog.currency.notFound');
			}

			$this['meta']
				->setMeta('title', $productDao->getName())
				->setMeta('og:title', $productDao->getName(), 'property')
				->setMeta('og:type', 'og:product', 'property')
				->setMeta('og:description', Strings::truncate(strip_tags((string) $productDao->getDescription()), 255), 'property')
				->setMeta('og:url', $url, 'property')
				->setMeta('product:plural_title', $productDao->getName(), 'property')
				->setMeta('product:price:amount', (string) round($productDao->getPrice(), $currency->decimals), 'property')
				->setMeta('product:price:currency', $currency->getCode(), 'property')
				->setMeta('product:condition', ProductConditionEnum::toGoogle($productDao->condition), 'property');
			if ($productDao->getAvailability()) {
				$this['meta']->setMeta('product:availability', $productDao->getAvailability()->getOpenGraphText(), 'property');
			}
		}


		$templatePageId = $nav->componentParams['templatePage2'];
		$templatePage   = null;
		if ($templatePageId) {
			$templatePage = $this->templatePagesService->get((int) $templatePageId);

			if ($templatePage) {
				$texts = $templatePage->getTexts();
				$texts = $this->pagesHelper->checkTexts($texts);

				$this->template->templateTextValues = $texts;
				$this->template->setFile($this->pathsService->getTemplatePagesFile($templatePage->getTemplate()));
			}
		}

		if (!$templatePage) {
			$this->template->setFile(EditablePackage::getPackageTemplate($nav->componentParams['templatePage']));
		}
	}

	public function renderEditablePackage(): void
	{
	}
}
