<?php declare(strict_types = 1);

namespace EshopPackages\Model\Services;

use Core\Model\Helpers\Strings;
use EshopCatalog\FrontModule\Model\Categories;
use Core\Model\Entities\QueryBuilder;
use Core\Model\Sites;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\FrontModule\Model\ProductQuery;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopPackages\Model\Entities\EditablePackage;
use Contributte\Translation\Translator;
use Nette\Utils\Json;

class PackageProducts
{
	protected ProductsFacade $productsFacade;
	protected Translator $translator;
	protected Sites $sites;
	protected Categories $categories;

	public function __construct(ProductsFacade $productsFacade, Translator $translator, Sites $sites, Categories $categories)
	{
		$this->productsFacade = $productsFacade;
		$this->translator     = $translator;
		$this->sites          = $sites;
		$this->categories     = $categories;
	}

	public function getProductsFacade(): ProductsFacade { return $this->productsFacade; }

	public function getProductsForEditablePackage(EditablePackage $package, bool $get2 = false): array
	{
		$ids = [];

		if ($get2) {
			$inCategories      = $package->getCategories2();
			$withFeatures      = $package->getFeatures2();
			$withManufacturers = $package->getManufacturers2();
		} else {
			$inCategories      = $package->getCategories();
			$withFeatures      = $package->getFeatures();
			$withManufacturers = $package->getManufacturers();
		}

		$er = $this->productsFacade->productsService->getEr();

		$query = (new ProductQuery($this->translator->getLocale()))
			->selectIds()
			->inSite($this->sites->getCurrentSite()->getIdent());

		/** @var QueryBuilder $qb */
		$qb = $query->getQueryBuilder($er);

		if ($inCategories) {
			$qb->andWhere('sites.category IN (:cats)')
				->setParameter('cats', $inCategories);
		}

		if ($withFeatures) {
			$features = [];
			foreach ($withFeatures as $v) {
				$v          = explode('-', $v);
				$features[] = end($v);
			}

			$qb->innerJoin('p.featureProducts', 'f', Join::WITH, 'f.featureValue IN (:feas)')
				->setParameter('feas', $features);
		}

		if ($withManufacturers) {
			$qb->andWhere('p.manufacturer IN (:manu)')
				->setParameter('manu', $withManufacturers);
		}

		foreach ($qb->getQuery()->getArrayResult() as $row) {
			$ids[] = $row['id'];
		}

		$ids = array_unique($ids);

		$tmp  = [];
		$arr  = [];
		$cats = [];
		foreach ($this->productsFacade->getProducts($ids) as $prod) {
			if (!$prod->canAddToCart) {
				continue;
			}

			$c = $prod->defaultCategory;

			if (!$get2) {
				while ($c->getParent()) {
					$c = $c->getParent();
				}
			}

			$cats[$c->getId()]                = $c->name;
			$tmp[$c->getId()][$prod->getId()] = $prod;
		}

		foreach ($this->categories->getEr()->createQueryBuilder('c')
			         ->select('c.id')
			         ->where('c.id IN (:ids)')
			         ->setParameter('ids', array_keys($cats))
			         ->orderBy('c.root')->addOrderBy('c.lft')->getQuery()->getArrayResult() as $row) {
			if (!isset($tmp[$row['id']])) {
				continue;
			}

			$arr[$cats[$row['id']]] = $tmp[$row['id']];
		}

		// Sort
		$packageSort = $package->getAttr('productsSort') ? Json::decode($package->getAttr('productsSort'), Json::FORCE_ARRAY) : [];

		foreach ($arr as $name => $rows) {
			$camelCaseName = Strings::toCamelCase($name);
			$products      = [];

			$ordered = [];
			foreach ($packageSort[$camelCaseName] as $key) {
				if (isset($rows[$key])) {
					$ordered[$key] = $rows[$key];
					unset($rows[$key]);
				}
			}

			$arr[$name] = $ordered + $rows;
		}

		return $arr;
	}
}
