<?php declare(strict_types = 1);

namespace EshopPayments\AdminModule\Components;

use Core\Model\Helpers\Strings;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Currency\Model\Entities\Currency;
use EshopCatalog\Model\Entities\SellerBankAccount;
use EshopPayments\Model\Entities\Payment;
use EshopPayments\Model\Entities\Status;
use EshopPayments\Model\Facade\PaymentsFacade;
use EshopPayments\Model\Repository\Payments;
use Nette\Utils\Html;

class PaymentsOverviewGrid extends BaseControl
{
	protected SellerBankAccount $account;
	protected Payments $payments;
	protected PaymentsFacade $paymentsFacade;

	public function __construct(SellerBankAccount $account, Payments $payments, PaymentsFacade $paymentsFacade)
	{
		$this->account = $account;
		$this->payments = $payments;
		$this->paymentsFacade = $paymentsFacade;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	public function createComponentGrid(): BaseDataGrid
	{
		$currencies = $this->getCurrencyPairs();

		$qb = $this->payments->getEr()->createQueryBuilder('p');
		$qb->addSelect('a, o')
		   ->join('p.bankAccount', 'a')
		   ->join('p.status', 's')
		   ->leftJoin('p.order', 'o')
		   ->andWhere($qb->expr()->eq('a.id', $this->account->getId()));

		$grid = $this->createGrid();
		$grid->setDataSource($qb);

		$grid->setDefaultSort(['dateOfPayment' => 'DESC']);

		$grid->addColumnText('order', 'eshopOrders.default.orderNumberShort')->setRenderer(function(Payment $p) {
			if ($p->order !== null) {
				return Html::el('a', [
					'target' => '_blank',
					'href'   => $this->presenter->link(':EshopOrders:Admin:Default:editOrder', $p->order->getId())
				])->setText($p->order->getId() ?? '');
			}
			return '';
		})->setFitContent()
			 ->setAlign('center');
		$grid->addColumnDateTime('dateOfPayment', 'eshopPayments.paymentsOverviewGrid.dateOfPayment')
			 ->setSortable()
			 ->setFitContent();
		$grid->addColumnText('senderAccount', 'eshopPayments.paymentsOverviewGrid.senderAccount')
			 ->setRenderer(fn(Payment $p) => !empty($p->senderAccountNumber) && !empty($p->senderBankCode) ? "{$p->senderAccountNumber} / {$p->senderBankCode}" : $p->senderAccountNumber);
		$grid->addColumnText('amount', 'eshopPayments.paymentsOverviewGrid.amount')
			 ->setRenderer(fn(Payment $p) => Strings::priceFormat((float) $p->amount, $currencies[strtolower($p->bankAccount->currencyCode)] ?? ''));
		$grid->addColumnText('variableSymbol', 'eshopPayments.paymentsOverviewGrid.variableSymbol')
			 ->setFitContent()
			 ->setSortable()
			 ->setFilterText();
		$grid->addColumnText('constantSymbol', 'eshopPayments.paymentsOverviewGrid.constantSymbol')
			 ->setFitContent()
			 ->setFilterText();
		$grid->addColumnText('specificSymbol', 'eshopPayments.paymentsOverviewGrid.specificSymbol')
			 ->setFitContent()
			 ->setFilterText();
		$grid->addColumnText('note', 'eshopPayments.paymentsOverviewGrid.note')
			 ->setTemplateEscaping(false)
			 ->setRenderer(function(Payment $p) {
				 if ($p->status->ident === Status::UNSYNCED && $p->order === null) {
					 return Html::el('span', ['class' => 'text-danger'])
								->setText($this->translator->translate('eshopPayments.paymentsOverviewGrid.messages.variableSymbolNotExists'));
				 } else {
					 if ($p->order !== null) {
						 $address = $p->order->getAddressInvoice();

						 if (!$address) {
							 return '';
						 }

						 $text = '';
						 if (!empty($address->getFirstName()) && !empty($address->getLastName())) {
							 $text = $address->getFirstName() . ' ' . $address->getLastName();
						 }
						 if (!empty($address->getCompany())) {
							 $text .= sprintf('(%s)', $address->getCompany());
						 }

						 return Html::el('span', ['class' => 'text-bold'])->setText($text);
					 }
				 }
			 });
		$grid->addColumnStatus('status', 'eshopPayments.paymentsOverviewGrid.status.title', 'status.ident')
			 ->setAlign('center')
			 ->setTemplate(__DIR__ . '/statusColumn.latte')
			 ->addOption('syncedAuto', 'eshopPayments.paymentsOverviewGrid.status.options.syncedAuto')->setIcon('check')
			 ->setClass('btn-success')->setShowTitle(true)->endOption()
			 ->addOption('syncedManual', 'eshopPayments.paymentsOverviewGrid.status.options.syncedManual')
			 ->setIcon('check')->setClass('btn-primary')->setShowTitle(true)->endOption()
			 ->addOption('unsynced', 'eshopPayments.paymentsOverviewGrid.status.options.unsynced')->setIcon('times')
			 ->setClass('btn-danger')->setShowTitle(true)->endOption()
			 ->addOption('returned', 'eshopPayments.paymentsOverviewGrid.status.options.returned')->endOption()
			->onChange[] = [$this, 'changeStatus'];

		// Filters
		$grid->addFilterSelect('status', '', [
			''             => '',
			'syncedAuto'   => $this->t('eshopPayments.paymentsOverviewGrid.status.options.syncedAuto'),
			'syncedManual' => $this->t('eshopPayments.paymentsOverviewGrid.status.options.syncedManual'),
			'unsynced'     => $this->t('eshopPayments.paymentsOverviewGrid.status.options.unsynced'),
			'returned'     => $this->t('eshopPayments.paymentsOverviewGrid.status.options.returned'),
		], 'status');

		return $grid;
	}

	/**
	 * @return string[]
	 */
	protected function getCurrencyPairs(): array
	{
		$result = [];
		foreach ($this->em->getRepository(Currency::class)->findPairs([], 'symbol', [], 'code') as $k => $v) {
			$result[strtolower((string) $k)] = $v;
		}

		return $result;
	}

	/**
	 * @param int|string $id
	 */
	public function changeStatus($id, string $newStatus): void
	{
		$presenter = $this->getPresenter();

		if (!$presenter) {
			return;
		}

		if ($this->paymentsFacade->changeStatus((int) $id, $newStatus)) {
			$presenter->flashMessageSuccess('eshopPayments.paymentsOverviewGrid.messages.statusChanged');
		} else {
			$presenter->flashMessageDanger('eshopPayments.paymentsOverviewGrid.messages.statusNotChanged');
		}

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

}