<?php declare(strict_types = 1);

namespace EshopPayments\Model\Facade;

use Core\Model\Application\AppState;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\Strings;
use Currency\AdminModule\Model\Currencies;
use Doctrine\ORM\AbstractQuery;
use EshopCatalog\Model\Entities\SellerBankAccount;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Invoices;
use EshopOrders\Model\Orders;
use EshopPayments\AdminModule\Model\Events\PaymentSyncEvent;
use EshopPayments\Model\Entities\IPayment;
use EshopPayments\Model\Entities\Payment;
use EshopPayments\Model\Entities\Status;
use EshopPayments\Model\Repository\Payments;
use EshopPayments\Model\Repository\Statuses;
use EshopPayments\Model\Utils\Numbers;
use Nette\Utils\Floats;
use Tracy\Debugger;

class PaymentsFacade
{
	protected EventDispatcher        $eventDispatcher;
	protected Orders                 $orders;
	protected Payments               $payments;
	protected Statuses               $statuses;
	protected EntityManagerDecorator $em;
	protected Currencies             $currencies;
	protected Invoices               $invoices;

	public function __construct(
		EventDispatcher        $eventDispatcher,
		Orders                 $orders,
		Payments               $payments,
		Statuses               $statuses,
		EntityManagerDecorator $em,
		Currencies             $currencies,
		Invoices               $invoices
	)
	{
		$this->eventDispatcher = $eventDispatcher;
		$this->orders          = $orders;
		$this->payments        = $payments;
		$this->statuses        = $statuses;
		$this->em              = $em;
		$this->currencies      = $currencies;
		$this->invoices        = $invoices;
	}

	public function sync(): void
	{
		$statuses = $this->statuses->getAssoc();
		$flush    = [];

		if (!$statuses) {
			Debugger::log("Fill table eshop_payment__status", 'eshopPayments');

			return;
		}

		$this->em->beginTransaction();
		try {
			/** @var IPayment[] $payments */
			$payments = [];

			// zde se jen ziskaji platby za dany ucet od posledni platby, ktera byla synchronizovana
			/** @var SellerBankAccount $account */
			foreach ($this->em->getRepository(SellerBankAccount::class)->findAll() as $account) {

				// synchronizuji jen ty ucty, ktere maji vyplneny cislo uctu a kod banky
				if (empty($account->numberPart1) || empty($account->numberPart2)) {
					continue;
				}

				// posledni platba za dany ucet (abych vedel odkud zacit synchronizaci)
				$lastSyncedPayment = $this->getLastPaymentByAccount($account->getId());

				$event = new PaymentSyncEvent($account, $lastSyncedPayment);
				$this->eventDispatcher->dispatch($event, self::class . '::onSync');
				$payments = array_merge($payments, $event->paymentsToSync);
			}

			// zde se provadi samotna synchronizace
			foreach ($payments as $payment) {

				// platby se zapornou castkou, jsou odchozi -> nechceme
				if ($payment->getAmount() < 0) {
					continue;
				}

				// overeni, ze neni platba z bank. uctu jeste v Payment tabulce (dle unikatniho id platby v db banky)
				if (count($this->payments->getBy($payment->getPaymentOrderId())) > 0) {
					continue;
				}

				$qb = $this->em->getRepository(Order::class)->createQueryBuilder('o');
				$qb->leftJoin('o.invoice', 'i')
					->where('o.isPaid = 0 AND (o.id = :id OR i.ident = :ident)')
					->setMaxResults(1)
					->setParameters([
						'id'    => (int) $payment->getVariableSymbol(),
						'ident' => $payment->getVariableSymbol(),
					]);

				/** @var Order|null $order */
				$order = $qb->getQuery()->getOneOrNullResult();

				$p                      = new Payment(Strings::formatEntityDecimal((string) $payment->getAmount()), $payment->getPaymentOrderId(), $payment->getBankAccount());
				$p->dateOfPayment       = $payment->getDateOfPayment();
				$p->senderAccountNumber = $payment->getSenderAccountNumber();
				$p->senderBankCode      = $payment->getSenderBankCode();
				$p->senderBankName      = $payment->getSenderBankName();
				$p->senderName          = $payment->getSenderName();
				$p->constantSymbol      = $payment->getConstantSymbol();
				$p->variableSymbol      = $payment->getVariableSymbol();
				$p->specificSymbol      = $payment->getSpecificSymbol();
				$p->userMessage         = $payment->getUserMessage();
				$p->comment             = $payment->getComment();

				if ($order === null) {
					$p->status = $statuses[Status::UNSYNCED];
				} else {
					$dec          = 2;
					$isPriceEqual = false;

					$invoice = $order->getInvoice();
					if ($invoice) {
						$this->invoices->setPriceCalculator($invoice);

						$defaultInvoicePrice = $invoice->invoiceData->getPrice();
					} else {
						$defaultInvoicePrice = -1;
					}

					$invoicePrice      = Numbers::keepDecimals($defaultInvoicePrice, $dec); // cena faktury na 2 desetinna mista
					$defaultOrderPrice = $order->getPrice(true);
					$orderPrice        = Numbers::keepDecimals($defaultOrderPrice, $dec); // cenu objednavky na 2 desetinna mista

					if (Floats::areEqual($orderPrice, $payment->getAmount()) || Floats::areEqual($invoicePrice, $payment->getAmount())) {
						$isPriceEqual = true;
					} else { // zkouska zda nebude sedet zaokrouhlena cena
						$currencyCode = $order->getCurrencyCode();
						$qb           = $this->currencies->getEr()->createQueryBuilder('c');
						/** @var array{decimals: int}|null $decimals */
						$decimals = $qb->select('c.decimals')
							->where('c.code = :codeUpper OR c.code = :codeLower')
							->setParameters([
								'codeUpper' => strtoupper($currencyCode),
								'codeLower' => strtolower($currencyCode),
							])
							->setMaxResults(1)
							->getQuery()->getOneOrNullResult(AbstractQuery::HYDRATE_ARRAY);
						if ($decimals !== null) {
							$decimals = $decimals['decimals'];
						}

						if ($decimals !== null &&
							(
								Floats::areEqual(round($orderPrice, $decimals), $payment->getAmount())
								|| Floats::areEqual(round($invoicePrice, $decimals), $payment->getAmount())
								|| Floats::areEqual(round($defaultOrderPrice, $decimals), $payment->getAmount())
								|| Floats::areEqual(round($defaultInvoicePrice, $decimals), $payment->getAmount())
							)) {
							$isPriceEqual = true;
						}
					}

					// pokud je objednavka nenalezena nebo nesedi castka, platba neni sparovana
					if ($isPriceEqual) {
						// sparovano (prepne se stav platby, spojeni s objednavkou, objednavka zaplacena)
						AppState::setState('eshopOrdersPaymentMethod', 'eshopOrders.paymentMethod.pairing');
						$p->status     = $statuses[Status::SYNCED_AUTO];
						$p->order      = $order;
						$order->isPaid = 1;

						$this->em->persist($order);
						$flush[] = $order;
					} else { // nesparovano
						$p->status = $statuses[Status::UNSYNCED];
					}
				}

				$this->em->persist($p);
				$flush[] = $p;

				$this->em->flush($flush);
			}

			$this->em->commit();
		} catch (\Exception $ex) {
			Debugger::log($ex, 'eshopPayments');
			if ($this->em->getConnection()->isTransactionActive()) {
				$this->em->getConnection()->rollBack();
			}
		}
	}

	public function changeStatus(int $id, string $newStatus): bool
	{
		try {
			/** @var Payment|null $item */
			$item   = $this->payments->getReference($id);
			$status = $this->statuses->get($newStatus);
			if ($item && $status) {
				$item->status = $status;
				$this->em->persist($item);
				$this->em->flush();

				return true;
			}

			return false;
		} catch (\Exception $exception) {
			Debugger::log($exception, 'eshopPayments');

			return false;
		}
	}

	protected function getLastPaymentByAccount(int $sellerBankAccountId): ?Payment
	{
		$qb = $this->payments->getEr()->createQueryBuilder('p');
		$qb->addSelect('a')
			->join('p.bankAccount', 'a')
			->where($qb->expr()->eq('a.id', $sellerBankAccountId))
			->addOrderBy('p.inserted', 'desc')
			->setMaxResults(1);

		return $qb->getQuery()->getOneOrNullResult();
	}

}
