<?php declare(strict_types = 1);

namespace EshopPrices\AdminModule\Components;

use Core\AdminModule\Model\Sites;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\AdminModule\Model\Categories;
use EshopCatalog\AdminModule\Model\Features;
use EshopCatalog\AdminModule\Model\FeatureValues;
use EshopCatalog\AdminModule\Model\Manufacturers;
use EshopCatalog\FrontModule\Model\CacheService;
use EshopOrders\AdminModule\Model\GroupsCustomers;
use EshopPrices\AdminModule\Model\Discounts;
use EshopPrices\Model\Entities\Discount;
use EshopPrices\Model\EshopPricesConfig;
use Exception;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;

class DiscountForm extends BaseControl
{
	/** @var int|null @persistent */
	public ?int $id = null;

	protected Discounts       $discounts;
	protected Features        $featuresService;
	protected FeatureValues   $featureValuesService;
	protected CacheService    $cacheService;
	protected Manufacturers   $manufacturers;
	protected Categories      $categories;
	protected Sites           $sites;
	protected GroupsCustomers $customersGroups;

	public function __construct(
		Discounts       $discounts,
		CacheService    $cacheService,
		Sites           $sites,
		Features        $features,
		FeatureValues   $featureValues,
		Manufacturers   $manufacturers,
		Categories      $categories,
		GroupsCustomers $customersGroups
	)
	{
		$this->discounts            = $discounts;
		$this->cacheService         = $cacheService;
		$this->sites                = $sites;
		$this->featuresService      = $features;
		$this->featureValuesService = $featureValues;
		$this->manufacturers        = $manufacturers;
		$this->categories           = $categories;
		$this->customersGroups      = $customersGroups;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();
		$form->getElementPrototype()->setAttribute('formnovalidate', true);

		$titlePrefix = 'eshopPrices.discount.';
		$form->addText('title', $titlePrefix . 'title');
		$form->addBool('isActive', 'default.isActive')
			->setDefaultValue(1);
		$form->addText('discount', $titlePrefix . 'discount')
			->setDefaultValue(0);

		// Skupiny zakazniku
		$groups = $this->customersGroups->getOptionsForSelect();
		$form->addCheckboxList('customerGroups', $titlePrefix . 'customerGroups', $groups);

		// Categories
		$sites = $this->sites->getAll();
		if (count($sites) > 1) {
			$flat = [];
			foreach ($sites as $site) {
				$tmp = $this->categories->getFlatTree($site->getIdent());

				if (empty($tmp)) {
					continue;
				}

				$flat[] = [
					'id'     => $tmp[0]['parent'],
					'parent' => 0,
					'name'   => $site->getIdent(),
				];

				$flat = array_merge($flat, $tmp);
			}
		} else {
			$flat = $this->categories->getFlatTree();
		}

		$form->addCheckboxNestedList('categories', $titlePrefix . 'categories', $flat);

		// Manufacturers
		$form->addCheckboxList('manufacturers', $titlePrefix . 'manufacturers', $this->manufacturers->getOptionsForSelect());

		// Features
		if (EshopPricesConfig::load('allowFeatures')) {
			$flat         = [];
			$featureTexts = $this->featuresService->getOptionsForSelect();
			foreach ($this->featureValuesService->getOptionsForSelect() as $featureId => $values) {
				$flat[] = [
					'id'     => 'f' . $featureId,
					'name'   => $featureTexts[$featureId],
					'parent' => 0,
				];

				foreach ($values as $valueId => $value) {
					$flat[] = [
						'id'     => $featureId . '-' . $valueId,
						'name'   => $value,
						'parent' => 'f' . $featureId,
					];
				}
			}
			$form->addCheckboxNestedList('params', $titlePrefix . 'features', $flat);
		}

		$form->addSaveCancelControl();

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			if ($this->id) {
				$discount = $this->discounts->get($this->id);
			} else {
				$discount = new Discount($values->discount ?: 0);
			}

			$discount->setDiscount($values->discount ?: 0);
			$discount->title    = $values->title;
			$discount->isActive = (int) $values->isActive;

			$features = [];
			if (EshopPricesConfig::load('allowFeatures')) {
				foreach ($values->params as $v) {
					if (substr($v, 0, 1) === 'f') {
						$features[] = $v;
					} else {
						$tmp        = explode('-', $v);
						$features[] = $tmp[1];
					}
				}
			}

			$discount->setManufacturers($values->manufacturers);
			$discount->setCategories($values->categories);
			$discount->setFeatures($features);
			$discount->setCustomerGroups($values->customerGroups);

			$this->em->persist($discount);
			$this->em->flush();

			$form->addCustomData('discountId', $discount->getId());
			$this->presenter->flashMessageSuccess('default.saved');
		} catch (Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	public function setDiscount(int $id): void
	{
		$this->id = $id;
		$discount = $this->discounts->get($id);

		if (!$discount) {
			throw new InvalidArgumentException;
		}


		$d = [
			'title'          => $discount->title,
			'discount'       => $discount->getDiscount(),
			'features'       => EshopPricesConfig::load('allowFeatures') ? $discount->getFeatures() : [],
			'manufacturers'  => $discount->getManufacturers(),
			'categories'     => $discount->getCategories(),
			'customerGroups' => $discount->getCustomerGroups(),
			'isActive'       => $discount->isActive,
		];

		$this['form']->setDefaults($d);
	}
}
