<?php declare(strict_types = 1);

namespace EshopPrices\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use EshopPrices\AdminModule\Model\Discounts;
use EshopPrices\Model\Entities\Discount;

class DiscountsGrid extends BaseControl
{
	protected Discounts $discounts;
	protected IDiscountFormFactory $formFactory;

	public function __construct(Discounts $discounts, IDiscountFormFactory $formFactory)
	{
		$this->discounts   = $discounts;
		$this->formFactory = $formFactory;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ============= Handle
	 */

	public function handleAdd(): void
	{
		$this->template->modalTitle       = $this->t('eshopPrices.title.addDiscount');
		$this->template->modal            = 'form';
		$this->template->modalDialogClass = 'modal-xxl';
		$this->redrawControl('modal');
	}

	public function handleEdit(int $id): void
	{
		$this['form']->id                 = $id;
		$this->template->modalTitle       = $this->t('eshopPrices.title.editDiscount');
		$this->template->modal            = 'form';
		$this->template->modalDialogClass = 'modal-xxl';
		$this->redrawControl('modal');
	}

	/**
	 * @param string|int|string[]|int[] $id
	 */
	public function handleDelete($id): void
	{
		$presenter = $this->presenter;
		$error     = false;

		if (is_array($id)) {
			$this->em->beginTransaction();
			foreach ($id as $v) {
				if (!$this->discounts->remove((int) $v)) {
					$error = true;
				}
			}
			if ($error) {
				$this->em->rollback();
			} else {
				$this->em->commit();
			}
		} else {
			$this->discounts->remove((int) $id);
		}

		if (!$error) {
			$presenter->flashMessageSuccess('default.removed');
		}
		else {
			$presenter->flashMessageDanger('default.removeFailed');
		}

		$presenter->redrawControl('flashes');
		$this['grid']->reload();
	}

	/*******************************************************************************************************************
	 * ============= Components
	 */

	public function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$qb = $this->discounts->getEr()->createQueryBuilder('d');

		$grid->setDataSource($qb);

		$columnPrefix = 'eshopPrices.discount.';

		// Columns
		$grid->addColumnText('title', $columnPrefix . 'title');
		$grid->addColumnNumber('discount', $columnPrefix . 'discount')->setRenderer(function(Discount $row) {
			return number_format($row->getDiscount(), 2, ',', ' ') . ' %';
		});

		// Action
		$grid->addAction('edit', '', 'edit!')->setIcon('edit')->setBsType('primary')
			->addClass('ajax');
		$grid->addAction('delete', '', 'delete!')->setIcon('times')->setBsType('danger')
			->addClass('ajax')
			->setConfirm('default.reallyDelete');

		$grid->addGroupAction('default.delete')->onSelect[] = [$this, 'handleDelete'];

		// Prototype
		$grid->getColumn('discount')->getElementPrototype('td')->class[] = 'w1nw';

		return $grid;
	}

	protected function createComponentForm(): DiscountForm
	{
		$control = $this->formFactory->create();

		if ($this->getParameter('id')) {
			$control->setDiscount((int) $this->getParameter('id'));
		}

		$control->onAnchor[] = function() use ($control) {
			$control['form']->onSuccessSave[] = function() {
				$this->getPresenter()->redrawControl('flashes');
				$this['grid']->reload();
			};

			$control['form']->onSuccessSaveAndClose[] = function() {
				$this->getPresenter()->payload->hideModal = true;
				$this->getPresenter()->redrawControl('flashes');
				$this['grid']->reload();
			};

			$control['form']->getComponent('saveControl')->closeModalOnCancel();
		};

		return $control;
	}
}
