<?php declare(strict_types = 1);

namespace EshopPrices\FrontModule\Model\Subscribers;

use Core\Model\Event\Event;
use Core\Model\Helpers\Strings;
use Currency\Model\Currencies;
use EshopCatalog\FrontModule\Model\Event\ProductsEvent;
use EshopCatalog\FrontModule\Model\Products;
use EshopOrders\FrontModule\Model\Customers;
use EshopPrices\FrontModule\Model\Discounts;
use EshopPrices\Model\EshopPricesConfig;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Users\Model\Security\User;

class ProductsSubscriber implements EventSubscriberInterface
{
	protected Discounts  $discounts;
	protected Currencies $currencies;
	protected User       $user;
	protected Customers  $customers;
	public static array  $used = [];

	public function __construct(
		Discounts  $discounts,
		Currencies $currencies,
		User       $user,
		Customers  $customers
	)
	{
		$this->discounts  = $discounts;
		$this->currencies = $currencies;
		$this->user       = $user;
		$this->customers  = $customers;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			Products::class . '::afterLoadFresh' => ['afterLoadFresh', 200],
			'eshopOrders.reloadCart'             => 'reloadCart',
		];
	}

	public function afterLoadFresh(ProductsEvent $event): void
	{
		$discounts = $this->discounts->getPublished();

		$defaultCurrency = $this->currencies->getAll()[$this->currencies->getDefaultCode()] ?? $this->currencies->getCurrent();
		$decimals        = $defaultCurrency->decimals;

		foreach ($event->products as &$product) {
			if (in_array($product->getId(), self::$used)) {
				continue;
			}

			foreach ($discounts as $discount) {
				$modifier = 1 - ($discount['discount'] / 100);

				if ($discount['customerGroups']) {
					if ($this->user->isLoggedIn()) {
						$customer = $this->customers->getByUserId($this->user->getId());
						if (!$customer->getGroupCustomers() || !in_array($customer->getGroupCustomers()->getId(), $discount['customerGroups'])) {
							continue;
						}
					} else {
						continue;
					}
				}

				if ($discount['manufacturers'] && !in_array((int) $product->manufacturerId, $discount['manufacturers'])) {
					continue;
				}

				if ($discount['categories']) {
					$has = false;

					foreach ($discount['categories'] as $dCat) {
						if ($dCat == $product->defaultCategoryId || in_array($dCat, (array) $product->categories)) {
							$has = true;
							break;
						}
					}

					if (!$has) {
						continue;
					}
				}

				if ($discount['features'] && EshopPricesConfig::load('allowFeatures')) {
					$has = false;

					if (!$product->getFeatures()) {
						continue;
					}

					foreach ($discount['features'] as $dFea) {
						if (Strings::startsWith((string) $dFea, 'f')) {
							foreach ($product->getFeatures() as $fea) {
								if ($fea->idFeature == substr($dFea, 1)) {
									$has = true;
									break 2;
								}
							}
						} else if (in_array($dFea, array_keys($product->getFeatures()))) {
							$has = true;
							break;
						}
					}

					if (!$has) {
						continue;
					}
				}

				$product->basePriceInBaseCurrency   = round($product->basePriceInBaseCurrency * $modifier, $decimals);
				$product->priceInBaseCurrency       = round($product->priceInBaseCurrency * $modifier, $decimals);
				$product->retailPriceInBaseCurrency = round($product->retailPriceInBaseCurrency * $modifier, $decimals);

				self::$used[] = $product->getId();
			}
		}
	}

	public function reloadCart(Event $event): void
	{
		self::$used = [];
	}
}
