<?php declare(strict_types = 1);

namespace EshopProductionWarehouse\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopProductionWarehouse\AdminModule\Model\Facade\ProductsFacade;
use EshopProductionWarehouse\AdminModule\Model\Facade\RawMaterialsFacade;
use EshopProductionWarehouse\AdminModule\Model\FormContainers\ProductsContainer\ProductsContainer;
use EshopProductionWarehouse\AdminModule\Model\FormContainers\ProductsContainer\ProductsContainerConfig;
use EshopProductionWarehouse\AdminModule\Presenters\BasePresenter;
use EshopProductionWarehouse\Model\Entities\Batch;
use EshopProductionWarehouse\Model\Entities\BatchInput;
use EshopProductionWarehouse\Model\Entities\BatchOutput;
use EshopProductionWarehouse\Model\Utils\UnitsHelper;
use Nette\Utils\ArrayHash;
use Tracy\Debugger;

/**
 * @property-read BasePresenter $presenter
 */
class BatchForm extends BaseControl
{
	protected RawMaterialsFacade $rawMaterialsFacade;
	protected ProductsFacade $productsFacade;
	protected ProductsContainer $productsContainer;
	protected array $rawMaterialItems;
	protected array $productItems;
	protected ?Batch $batch = null;
	protected const RAW_MATERIALS_TEXT_PREFIX = 'eshopProductionWarehouse.batchForm.rawMaterials';
	protected const PRODUCTS_TEXT_PREFIX = 'eshopProductionWarehouse.batchForm.products';

	public function __construct(RawMaterialsFacade $rawMaterialsFacade, ProductsFacade $productsFacade, ProductsContainer $productsContainer)
	{
		$this->rawMaterialsFacade = $rawMaterialsFacade;
		$this->productsFacade = $productsFacade;
		$this->rawMaterialItems = $this->rawMaterialsFacade->getAll();
		$this->productItems = $this->productsFacade->getFinalProducts();
		$this->productsContainer = $productsContainer;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	public function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setShowLangSwitcher(false)
			 ->setAjax();

		$form->addText('name', 'eshopProductionWarehouse.batchForm.name')
			 ->setRequired();

		$rawMaterialPairs = [];
		foreach ($this->rawMaterialItems as $item) {
			$rawMaterialPairs[$item['id']] = $item['name'];
		}

		$productItemPairs = [];
		foreach ($this->productItems as $item) {
			$productItemPairs[$item['id']] = $item['name'];
		}

		$rawMaterialUnits = [];
		foreach ($this->rawMaterialItems as $id => $rawMaterial) {
			$rawMaterialUnits[$id] = [
				'allowedUnits' => array_map(fn($v) => $this->t('eshopCatalog.productForm.units.items.' . $v), UnitsHelper::findRelatedUnits($rawMaterial['units'])),
				'unit'         => $rawMaterial['units'],
				'unitText'     => $this->t('eshopCatalog.productForm.units.items.' . $rawMaterial['units']),
			];
		}

		$productUnits = [];
		foreach ($this->productItems as $id => $product) {
			$productUnits[$id] = [
				'allowedUnits' => array_map(fn($v) => $this->t('eshopCatalog.productForm.units.items.' . $v), UnitsHelper::findRelatedUnits($product['units'])),
				'unit'         => $product['units'],
				'unitText'     => $this->t('eshopCatalog.productForm.units.items.' . $product['units']),
			];
		}

		$config = ProductsContainerConfig::create([ProductsContainerConfig::ALLOW_UNIT_CONVERSION => true]);
		if ($this->batch) {
			$this->productsContainer->createBulk($rawMaterialPairs, $rawMaterialUnits, self::RAW_MATERIALS_TEXT_PREFIX, $form, 'rawMaterials', $this->batch->inputs->toArray(), $config);
			$this->productsContainer->createBulk($productItemPairs, $productUnits, self::PRODUCTS_TEXT_PREFIX, $form, 'products', $this->batch->outputs->toArray(), $config);
		} else {
			$this->productsContainer->createOne($rawMaterialPairs, $rawMaterialUnits, self::RAW_MATERIALS_TEXT_PREFIX, $form, 'rawMaterials', $config);
			$this->productsContainer->createOne($productItemPairs, $productUnits, self::PRODUCTS_TEXT_PREFIX, $form, 'products', $config);
		}

		$form->addSaveCancelControl();
		$form['saveControl']['cancel']->setHtmlAttribute('onclick', sprintf('if (document.batchFormEdited) {if (confirm(\'%s\')) window.history.go(-1); else return false;} else window.history.go(-1);', $this->t('eshopProductionWarehouse.batchForm.askCancel')));

		$form->onValidate[] = [$this, 'formValidate'];
		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values): void
	{
		$this->productsContainer->validate(self::RAW_MATERIALS_TEXT_PREFIX, $form, 'rawMaterials');
		$this->productsContainer->validate(self::PRODUCTS_TEXT_PREFIX, $form, 'products');
		$this->redrawControl('formErrors');
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): void
	{
		try {
			$this->em->beginTransaction();

			if (!$this->batch) {
				$this->batch = new Batch($values->name);
			}

			$this->batch->name = $values->name;

			$newInputIds = [];
			$newOutputIds = [];
			$oldOutputIds = $this->batch->getOutputIds();
			$oldInputIds = $this->batch->getInputIds();

			foreach ($this->productsContainer->process($form, 'rawMaterials') as $rawMaterial) {
				$quantity = isset($rawMaterial->getExtras()['inUnit'])
					? UnitsHelper::convert($rawMaterial->getExtras()['inUnit'], $rawMaterial->getProduct()->units, $rawMaterial->getQuantity())
					: $rawMaterial->getQuantity();

				if ($rawMaterial->getId()) {
					/** @var BatchInput|null $input */
					$input = $this->em->getRepository(BatchInput::class)->find($rawMaterial->getId());
					$newInputIds[] = $rawMaterial->getId();
				} else {
					$input = new BatchInput($rawMaterial->getProduct(), $this->batch, $quantity);
					$this->batch->addInput($input);
				}

				$input->setProduct($rawMaterial->getProduct());
				$input->setQuantity($quantity);
			}

			foreach ($this->productsContainer->process($form, 'products') as $product) {
				$quantity = isset($product->getExtras()['inUnit'])
					? UnitsHelper::convert($product->getExtras()['inUnit'], $product->getProduct()->units, $product->getQuantity())
					: $product->getQuantity();

				if ($product->getId()) {
					/** @var BatchOutput|null $output */
					$output = $this->em->getRepository(BatchOutput::class)->find($product->getId());
					$newOutputIds[] = $product->getId();
				} else {
					$output = new BatchOutput($product->getProduct(), $this->batch, $quantity);
					$this->batch->addOutput($output);
				}

				$output->setProduct($product->getProduct());
				$output->setQuantity($quantity);
			}

			foreach (array_diff($oldInputIds, $newInputIds) as $id) {
				$input = $this->em->getRepository(BatchInput::class)->find((int) $id);
				if (!$input) {
					continue;
				}

				$this->batch->removeInput($input);
			}

			foreach (array_diff($oldOutputIds, $newOutputIds) as $id) {
				$output = $this->em->getRepository(BatchOutput::class)->find((int) $id);
				if (!$output) {
					continue;
				}

				$this->batch->removeOutput($output);
			}

			$this->em->persist($this->batch)->flush();

			$this->em->commit();

			$form->addCustomData('id', $this->batch->getId());

			$this->presenter->flashMessageSuccess('default.saved');
		} catch (\Exception $exception) {
			Debugger::log($exception);
			$this->em->rollback();
			$form->addError($exception->getMessage());
			$this->redrawControl('formErrors');
		}
	}

	public function setBatch(Batch $batch): void
	{
		$this->batch = $batch;
		$form = $this['form'];
		$form->setDefaults(['name' => $batch->name]);
	}

}