<?php declare(strict_types = 1);

namespace EshopProductionWarehouse\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\Form\BaseForm;
use EshopProductionWarehouse\AdminModule\Model\Facade\WarehouseFacade;
use EshopProductionWarehouse\AdminModule\Model\Repository\WarehouseRepository;
use Nette\Utils\ArrayHash;
use Nette\Utils\DateTime;
use Nette\Utils\Floats;
use Nette\Utils\Html;
use Nette\Utils\Validators;

class WarehouseToDateControl extends BaseControl
{
	protected WarehouseRepository $warehouseRepository;
	protected WarehouseFacade $warehouseFacade;

	public function __construct(WarehouseFacade $warehouseFacade, WarehouseRepository $warehouseRepository)
	{
		$this->warehouseFacade = $warehouseFacade;
		$this->warehouseRepository = $warehouseRepository;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	public function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setShowLangSwitcher(false);

		$form->addDatePicker('toDate', 'eshopProductionWarehouse.warehouseToDateControl.form.date')
			 ->setDefaultValue($this->getDateParam())
			 ->setRequired();
		$form->addSelect('warehouse', 'eshopProductionWarehouse.warehouseToDateControl.form.warehouse.caption', [
				null => 'eshopProductionWarehouse.warehouseToDateControl.form.warehouse.prompt'
			] + $this->warehouseRepository->findPairs())
			->setDefaultValue($this->getWarehouseIdParam());

		$form->addSubmit('submit', 'eshopProductionWarehouse.warehouseToDateControl.form.show');

		$form->onSuccess[] = function(BaseForm $form, ArrayHash $values) {
			$this->presenter->redirect('this', [
				'date'        => DateTime::from($values->toDate)->format('Y-m-d'),
				'warehouseId' => Validators::isNone($values->warehouse) ? null : (int) $values->warehouse
			]);
		};

		return $form;
	}

	public function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$grid->setDataSource($this->warehouseFacade->getState($this->getDateParam(), null, $this->getWarehouseIdParam()));
		$grid->addColumnLink('name', 'eshopProductionWarehouse.warehouseToDateControl.grid.name', ':EshopCatalog:Admin:Products:edit')
			 ->setFilterText();
		$grid->addColumnText('quantity', 'eshopProductionWarehouse.warehouseToDateControl.grid.quantity')
			 ->setFitContent()
			 ->setTemplateEscaping(false)
			 ->setRenderer(function(array $data) {
				 if ($data['unlimitedQuantity']) {
					 $data['quantity'] = '∞';
				 }

				 $text = sprintf('%s %s', $data['quantity'], $this->t('eshopCatalog.productForm.units.itemShortcuts.' . $data['units']));
				 if (!is_string($data['quantity']) && Floats::isLessThanOrEqualTo($data['quantity'], 0.0)) {
					 return Html::el('span', ['style' => 'color:red'])->setText($text);
				 }

				 return $text;
			 });

		return $grid;
	}

	protected function getDateParam(): ?DateTime
	{
		return $this->presenter->getParameter('date') ? DateTime::from($this->presenter->getParameter('date')) : new DateTime;
	}

	protected function getWarehouseIdParam(): ?int
	{
		$warehouseId = $this->presenter->getParameter('warehouseId');
		return $warehouseId ? ((int) $warehouseId) : null;
	}

}