<?php declare(strict_types = 1);

namespace EshopProductionWarehouse\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopProductionWarehouse\AdminModule\Model\Facade\ProductsFacade;
use EshopProductionWarehouse\AdminModule\Model\Facade\WarehouseFacade;
use EshopProductionWarehouse\AdminModule\Model\FormContainers\ProductsContainer\ProductsContainer;
use EshopProductionWarehouse\AdminModule\Model\FormContainers\ProductsContainer\ProductsContainerConfig;
use EshopProductionWarehouse\AdminModule\Model\Repository\WarehouseRepository;
use EshopProductionWarehouse\AdminModule\Presenters\BasePresenter;
use EshopProductionWarehouse\Model\Entities\Warehouse;
use EshopProductionWarehouse\Model\Entities\WarehouseMovement;
use EshopProductionWarehouse\Model\Entities\WarehouseMovementInput;
use EshopProductionWarehouse\Model\Entities\WarehouseMovementOutput;
use EshopProductionWarehouse\Model\Utils\UnitsHelper;
use EshopProductionWarehouse\Model\Utils\Validators;
use Nette\Utils\ArrayHash;
use Nette\Utils\DateTime;
use Tracy\Debugger;

/**
 * @property-read BasePresenter $presenter
 */
class WarehouseTransferForm extends BaseControl
{
	protected WarehouseFacade $warehouseFacade;
	protected WarehouseRepository $warehouseRepository;
	protected ProductsContainer $productsContainer;
	protected ProductsFacade $productsFacade;
	protected array $productItems;
	protected const PRODUCTS_TEXT_PREFIX = 'eshopProductionWarehouse.warehouseTransferForm.products';

	public function __construct(WarehouseFacade $warehouseFacade, ProductsFacade $productsFacade, WarehouseRepository $warehouseRepository, ProductsContainer $productsContainer)
	{
		$this->productsFacade = $productsFacade;
		$this->warehouseFacade = $warehouseFacade;
		$this->warehouseRepository = $warehouseRepository;
		$this->productItems = $this->productsFacade->getAll();
		$this->productsContainer = $productsContainer;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	public function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setShowLangSwitcher(false)
			 ->setAjax();

		$warehousePairs = $this->warehouseRepository->findPairs();

		$form->addDatePicker('date', 'eshopProductionWarehouse.warehouseTransferForm.date')
			 ->setDefaultValue(new DateTime)
			 ->setRequired();
		$form->addSelect('fromWarehouse', 'eshopProductionWarehouse.warehouseTransferForm.fromWarehouse', [null => null] + $warehousePairs)
			 ->setRequired();
		$form->addSelect('toWarehouse', 'eshopProductionWarehouse.warehouseTransferForm.toWarehouse', [null => null] + $warehousePairs)
			 ->setRequired()
			 ->addRule($form::NOT_EQUAL, 'eshopProductionWarehouse.warehouseTransferForm.errors.equalWarehouses', $form['fromWarehouse']);

		$productItemPairs = [];
		foreach ($this->productItems as $item) {
			$productItemPairs[$item['id']] = $item['name'];
		}

		foreach ($this->warehouseRepository->findPairs() as $wId => $w) {
			$products = $this->warehouseFacade->getState(null, array_keys($productItemPairs), $wId);
			foreach ($products as $p) {
				$text = sprintf('%s %s', $p['quantity'], $this->t('eshopCatalog.productForm.units.itemShortcuts.' . $p['units']));
				$productItemPairs[$p['id']] .= sprintf(
					' (%s: %s)',
					$w,
					$p['unlimitedQuantity'] ? '∞' : $text
				);
			}
		}

		$productUnits = [];
		foreach ($this->productItems as $id => $product) {
			$productUnits[$id] = [
				'allowedUnits' => array_map(fn($v) => $this->t('eshopCatalog.productForm.units.items.' . $v), UnitsHelper::findRelatedUnits($product['units'])),
				'unit'         => $product['units'],
				'unitText'     => $this->t('eshopCatalog.productForm.units.items.' . $product['units']),
			];
		}

		$config = ProductsContainerConfig::create();

		$this->productsContainer->createOne($productItemPairs, $productUnits, self::PRODUCTS_TEXT_PREFIX, $form, 'products', $config);

		$form->addSaveCancelControl();
		unset($form['saveControl']['save']);

		$form->onValidate[] = [$this, 'formValidate'];
		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values): void
	{
		$this->productsContainer->validate(self::PRODUCTS_TEXT_PREFIX, $form, 'products');

		$operations = $this->productsContainer->process($form, 'products');

		$products = [];
		foreach ($operations as $operation) {
			$products[$operation->getProduct()->getId()] = [
				'newQuantity' => $operation->getQuantity(),
				'diffQuantity' => -$operation->getQuantity()
			];
		}

		$warehouseState = $this->warehouseFacade->getState(null, array_keys($products), ($this->em->getRepository(Warehouse::class)->find((int) $values->fromWarehouse))->getId());

		foreach ($warehouseState as $warehouseProductState) {
			$change = $products[$warehouseProductState['id']];
			if (Validators::isDisallowedProductQuantity($warehouseProductState, $change['diffQuantity'])) {
				$form->addError($this->t('eshopProductionWarehouse.warehouseMovementForm.errors.negativeProductQuantity', null, ['product' => $warehouseProductState['name']]), false);
			}
		}

		$this->redrawControl('formErrors');
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): void
	{
		try {
			$this->em->beginTransaction();

			$products = $this->productsContainer->process($form, 'products');

			$warehouseMovement = new WarehouseMovement($values['date']);
			$warehouseMovement->warehouse = $this->warehouseRepository->get((int) $values->fromWarehouse);

			// vstupy dávky jsou úbytkem na sklade
			foreach ($products as $input) {
				$product = $input->getProduct();
				$quantity = $input->getQuantity();
				$warehouseMovementOutput = new WarehouseMovementOutput($product, $warehouseMovement, $quantity);
				$warehouseMovement->addOutput($warehouseMovementOutput);
			}

			$this->em->persist($warehouseMovement);

			$warehouseMovement = new WarehouseMovement($values['date']);
			$warehouseMovement->warehouse = $this->warehouseRepository->get((int) $values->toWarehouse);

			// vystupy dávky jsou prirustkem na sklade
			foreach ($products as $output) {
				$product = $output->getProduct();
				$quantity = $output->getQuantity();
				$warehouseMovementInput = new WarehouseMovementInput($product, $warehouseMovement, $quantity);
				$warehouseMovement->addInput($warehouseMovementInput);
			}

			$this->em->persist($warehouseMovement);

			$this->em->flush();
			$this->em->commit();

			$this->presenter->flashMessageSuccess('default.saved');

		} catch (\Exception $exception) {
			Debugger::log($exception);
			$this->em->rollback();
			$form->addError($exception->getMessage());
			$this->redrawControl('formErrors');
		}
	}

}