<?php declare(strict_types = 1);

namespace EshopProductionWarehouse\AdminModule\Model\Facade;

use Contributte\Translation\Translator;
use Core\Model\Entities\EntityManagerDecorator;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\Model\Entities\Product;
use EshopProductionWarehouse\AdminModule\Model\ProductExtraFields;
use EshopProductionWarehouse\Model\Entities\Warehouse;
use EshopProductionWarehouse\Model\Entities\WarehouseMovement;
use EshopProductionWarehouse\Model\Entities\WarehouseMovementInput;
use Nette\Utils\DateTime;

class WarehouseFacade
{
	protected EntityManagerDecorator $em;
	protected ProductExtraFields $productExtraFields;
	protected Translator $translator;

	public function __construct(EntityManagerDecorator $em, ProductExtraFields $productExtraFields, Translator $translator)
	{
		$this->em = $em;
		$this->productExtraFields = $productExtraFields;
		$this->translator = $translator;
	}

	/**
	 * @return array<array{id: int, name: string, quantity: float, unlimitedQuantity: bool, enabledNegativeQuantity: bool, units: string}>
	 */
	public function getState(?DateTime $toDate = null, ?array $productIds = null, ?int $warehouseId = null): array
	{
		$fnGetSumMovements = function(string $movement) use ($toDate, $productIds, $warehouseId) {
			$qb = $this->em->getRepository(WarehouseMovement::class)->createQueryBuilder('w');
			$qb->select('p.id, mpt.name, p.units, SUM(m.quantity) quantity, p.unlimitedQuantity')
			   ->addSelect(sprintf('(%s) as enabledNegativeQuantity', $this->productExtraFields->getEnabledNegativeQuantityQb()
																							   ->getDQL()))
			   ->join(sprintf('w.%s', $movement), 'm')
			   ->join('m.product', 'p')
			   ->join('p.productTexts', 'mpt', Join::WITH, 'mpt.lang = :lang')
			   ->groupBy('p')
			   ->setParameters([
				   'lang' => $this->translator->getLocale(),
				   'sn'   => Product::EXTRA_FIELD_SECTION,
				   'key2' => 'enabledNegativeQuantity'
			   ]);

			if ($warehouseId) {
				$qb->andWhere($qb->expr()->eq('w.warehouse', $warehouseId));
			}

			if ($toDate) {
				$qb->andWhere('w.date <= :date')
				   ->setParameter('date', $toDate->setTime(23, 59, 59));
			}

			if ($productIds) {
				$qb->andWhere($qb->expr()->in('p.id', $productIds));
			}

			return $qb->getQuery()->getArrayResult();
		};

		$inputs = array_map(static function(array $data) {
			$data['quantity'] = (float) $data['quantity'];
			return $data;
		}, $fnGetSumMovements('inputs'));
		$outputs = array_map(static function(array $data) {
			$data['quantity'] = ((float) $data['quantity']) * -1;
			return $data;
		}, $fnGetSumMovements('outputs'));

		$items = [];
		foreach ([$inputs, $outputs] as $arr) {
			foreach ($arr as $item) {
				$items[] = $item;
			}
		}

		$result = [];
		/** @var array{id: int, name: string, quantity: float, unlimitedQuantity: int, enabledNegativeQuantity: bool|null, units: string} $item */
		foreach ($items as $item) {
			$item['unlimitedQuantity'] = (bool) $item['unlimitedQuantity'];
			$item['enabledNegativeQuantity'] = (bool) $item['enabledNegativeQuantity'];
			if (!isset($result[$item['id']])) {
				$result[$item['id']] = $item;
			} else {
				$result[$item['id']]['quantity'] += $item['quantity'];
			}
		}

		return $result;
	}

	public function getAveragePurchasePrice(int $productId): float
	{
		$avg = 0;
		$qb = $this->em->getRepository(WarehouseMovementInput::class)->createQueryBuilder('i');
		$qb->select('i.price')
		   ->where('i.price IS NOT NULL AND i.product = :productId')
		   ->setParameter('productId', $productId);

		$items = array_map(static fn($item) => (float) $item['price'], $qb->getQuery()->getArrayResult());

		if ($items) {
			$sum = array_sum($items);
			$count = count($items);

			return $sum / $count;
		}

		return $avg;
	}

	public function getWarehouseByOrder(int $orderId): ?Warehouse
	{
		$qb = $this->em->getRepository(Warehouse::class)->createQueryBuilder('w');
		$qb->join('w.orders', 'wo')
		   ->join('wo.order', 'o')
		   ->where('o.id = :oId')
		   ->setParameter('oId', $orderId)
		   ->setMaxResults(1);

		return $qb->getQuery()->getOneOrNullResult();
	}

}