<?php declare(strict_types = 1);

namespace EshopProductionWarehouse\AdminModule\Model\FormContainers\ProductsContainer;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\Model\Entities\Product;
use EshopProductionWarehouse\AdminModule\Model\Common\IOperation;
use EshopProductionWarehouse\Model\Utils\UnitsHelper;
use Nette\Localization\Translator;
use Nette\Utils\Arrays;
use Nette\Utils\Strings;
use Nette\Utils\Validators;

class ProductsContainer
{
	protected Translator $translator;
	protected EntityManagerDecorator $em;
	protected const IDENT_PATTERN = '%s_%s__ident__';
	protected static ?string $whichIdHasAssets = null;

	public function __construct(Translator $translator, EntityManagerDecorator $em)
	{
		$this->translator = $translator;
		$this->em = $em;
	}

	/**
	 * @param array<int, string> $productPairs
	 */
	protected function createContainer(array $productPairs, string $textsTransPrefix, string $id, int $rowId, ?IOperation $operation = null, ProductsContainerConfig $config): BaseContainer
	{
		$productContainer = new BaseContainer;
		$productContainer->addSelect('product', $textsTransPrefix . '.list.productCaption', ['' => ''] + $productPairs)
						 ->setDefaultValue($operation ? $operation->getProduct()->getId() : null)
						 ->getControlPrototype()->setAttribute('data-source', $id);
		$productContainer->addText('quantity', $textsTransPrefix . '.list.quantityCaption')
						 ->setDefaultValue($operation ? $operation->getQuantity() : null);
		$productContainer->addHidden('id')
						 ->setDefaultValue($operation ? $operation->getId() : null)
						 ->setNullable();
		if ($config->allowUnitConversion) {
			$productContainer->addSelect(
				'inUnit',
				null,
				$operation ? array_map(fn($v) => $this->translator->translate('eshopCatalog.productForm.units.items.' . $v), UnitsHelper::findRelatedUnits($operation->getProduct()->units)) : null
			)->setDefaultValue($operation ? $operation->getProduct()->units : null);
		}

		if ($config->allowPurchasePrice) {
			$productContainer->addText('price')
							 ->setDefaultValue($operation ? $operation->getExtras()['price'] : null)
							 ->setNullable();
		}

		if ($operation) {
			$productContainer->addCustomData('unit', $this->translator->translate('eshopCatalog.productForm.units.items.' . $operation->getProduct()->units));
			$productContainer->addCustomData('averagePrice', $operation->getExtras()['averagePrice']);
		}

		return $productContainer;
	}


	/**
	 * @param array<int, array<string, mixed>> $unitPairs
	 */
	protected function createParentContainer(array $unitPairs, string $textsTransPrefix, BaseForm $form, string $id, ProductsContainerConfig $config): BaseContainer
	{
		$container = $form->addContainer($id);
		$container->addCustomData('template', __DIR__ . '/ProductsContainer.latte');

		if (!static::$whichIdHasAssets) {
			static::$whichIdHasAssets = $id;
			$form->addCustomData('whichIdHasAssets', $id);
		}

		$container->addCustomData('units', $unitPairs);
		$container->addCustomData('textsTransPrefix', $textsTransPrefix);
		$container->addCustomData('template', __DIR__ . '/ProductsContainer.latte');
		$container->addCustomData('config', $config);

		return $container;
	}

	protected function makeErrorMessage(?int $rowNumber, ?string $prefix, ?string $fieldName, string $messageTransPath): string
	{
		$prefix = $this->translator->translate($prefix) ? $prefix : null;
		$message = $prefix ? $this->translator->translate($prefix) . ': ' : '';
		$message .= str_replace('%label', $fieldName ? $this->translator->translate($fieldName) : '', $this->translator->translate($messageTransPath));
		return $message .= $rowNumber ? sprintf(' (%s %s)', $this->translator->translate('eshopProductionWarehouse.common.row'), $rowNumber) : '';
	}

	/**
	 * @param array<int, string> $productPairs
	 * @param array<int, array<string, mixed>> $unitPairs
	 */
	public function createOne(array $productPairs, array $unitPairs, string $textsTransPrefix, BaseForm $form, string $id, ProductsContainerConfig $config): void
	{
		$rowId = 1;
		$parentContainer = $this->createParentContainer($unitPairs, $textsTransPrefix, $form, $id, $config);
		$productContainer = $this->createContainer($productPairs, $textsTransPrefix, $id, $rowId, null, $config);
		$parentContainer->addComponent($productContainer, sprintf(self::IDENT_PATTERN, $id, $rowId));
	}

	/**
	 * @param array<int, string> $productPairs
	 * @param array<int, array<string, mixed>> $unitPairs
	 * @param IOperation[] $operations
	 */
	public function createBulk(array $productPairs, array $unitPairs, string $textsTransPrefix, BaseForm $form, string $id, array $operations, ProductsContainerConfig $config): void
	{
		$parentContainer = $this->createParentContainer($unitPairs, $textsTransPrefix, $form, $id, $config);

		foreach ($operations as $rowId => $operation) {
			$productContainer = $this->createContainer($productPairs, $textsTransPrefix, $id, $rowId, $operation, $config);
			$parentContainer->addComponent($productContainer, sprintf(self::IDENT_PATTERN, $id, $rowId));
		}
	}

	public function validate(string $textsTransPrefix, BaseForm $form, string $id): void
	{
		$values = $form->getValues();
		foreach ($values[$id] as $k => $p) {
			if ($form[$id][$k]) {
				foreach ($form[$id][$k]->getComponents() as $c) {
					$c->cleanErrors();
				}
			}
		}

		$vals = $form->getHttpData();
		foreach ($vals as $k => $v) {
			if (Strings::startsWith($k, $id)) { // validace tabulky produktu
				if (!$vals[$k]) {
					$form->addError($textsTransPrefix . '.errors.productsNotFilled');
					return;
				}

				$i = 0;
				$productsCounter = []; // pocet vyskytu produktu v tabulce produktu
				foreach ($vals[$k] as $row) {
					$i++;
					foreach ($row as $fieldName => $fieldValue) {
						switch ($fieldName) {
							case 'product':
								if (!isset($productsCounter[$fieldValue])) {
									$productsCounter[$fieldValue] = 1;
								} else {
									$productsCounter[$fieldValue]++;
								}

								if (Validators::isNone($fieldValue)) {
									$form->addError($this->makeErrorMessage(
										$i, $textsTransPrefix . '.errorHint',
										$textsTransPrefix . '.list.' . $fieldName,
										'default.formMessages.filled'
									), false);
								}
								break;
							case 'quantity':
								if (!((float) \Core\Model\Helpers\Strings::formatEntityDecimal($fieldValue) > 0)) {
									$form->addError($this->makeErrorMessage(
										$i, $textsTransPrefix . '.errorHint', $textsTransPrefix . '.list.' . $fieldName,
										$textsTransPrefix . '.errors.numberAndGreaterThan'
									), false);
								}
								break;
						}
					}
				}

				if (Arrays::some($productsCounter, static fn($count) => $count > 1)) {
					$form->addError($this->makeErrorMessage(
						null, null, null,
						$textsTransPrefix . '.errors.productUsedMoreTimes'
					), false);
				}
			}
		}
	}

	/**
	 * @return IOperation[]
	 */
	public function process(BaseForm $form, string $id): array
	{
		$result = [];
		$vals = $form->getHttpData();
		foreach ($vals as $k => $v) {
			if (Strings::startsWith($k, $id)) {
				foreach ($vals[$k] as $row) {
					$newRow = $row;
					$newRow['id'] = Validators::isNone($newRow['id']) ? null : (int) $newRow['id'];
					$newRow['product'] = (int) $newRow['product'];
					$newRow['quantity'] = (float) \Core\Model\Helpers\Strings::formatEntityDecimal($newRow['quantity']);

					/** @var Product|null $product */
					$product = $this->em->getRepository(Product::class)->find($newRow['product']);

					if (!$product) {
						continue;
					}

					$operation = new class($newRow['id'], $product, $newRow['quantity']) implements IOperation {

						protected ?int $id;
						protected Product $product;
						protected float $quantity;
						protected array $extras = [];

						public function __construct(?int $id, Product $product, float $quantity)
						{
							$this->id = $id;
							$this->product = $product;
							$this->quantity = $quantity;
						}

						public function getQuantity(): float
						{
							return $this->quantity;
						}

						public function setQuantity($quantity): self
						{
							$this->quantity = $quantity;
							return $this;
						}

						public function getProduct(): Product
						{
							return $this->product;
						}

						public function setProduct(Product $product): self
						{
							$this->product = $product;
							return $this;
						}

						public function getId(): ?int
						{
							return $this->id;
						}

						public function getExtras(): array
						{
							return $this->extras;
						}

						public function setExtras(array $extras): self
						{
							$this->extras = $extras;
							return $this;
						}

						public function setExtra($key, $val): self
						{
							$this->extras[$key] = $val;
							return $this;
						}

					};

					if (isset($newRow['inUnit']) && !Validators::isNone($newRow['inUnit'])) {
						$operation->setExtras(['inUnit' => $newRow['inUnit']]);
					}
					if (isset($newRow['price']) && !Validators::isNone($newRow['price'])) {
						$operation->setExtra('price', $newRow['price']);
					}

					$result[] = $operation;
				}
			}
		}

		return $result;
	}

}