<?php declare(strict_types = 1);

namespace EshopProductionWarehouse\AdminModule\Model\Subscribers;

use Contributte\Translation\Translator;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Entities\ExtraField;
use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\FormValidateEvent;
use Core\Model\Event\SetFormDataEvent;
use Core\Model\UI\Form\BaseForm;
use Core\Model\UI\Form\BootstrapRenderer;
use EshopCatalog\AdminModule\Components\Products\ProductForm;
use EshopCatalog\Model\Entities\Product;
use EshopProductionWarehouse\AdminModule\Model\Facade\WarehouseFacade;
use EshopProductionWarehouse\AdminModule\Model\Repository\WarehouseRepository;
use EshopProductionWarehouse\Model\Entities\WarehouseMovement;
use EshopProductionWarehouse\Model\Entities\WarehouseMovementInput;
use EshopProductionWarehouse\Model\EshopProductionWarehouseConfig;
use Nette\Utils\ArrayHash;
use Nette\Utils\DateTime;
use Nette\Utils\Floats;
use Nette\Utils\Strings;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class ProductFormSubscriber implements EventSubscriberInterface
{
	/** @var array<int, string> */
	protected array $warehousePairs = [];
	protected EntityManagerDecorator $em;
	protected WarehouseFacade $warehouseFacade;
	protected WarehouseRepository $warehouseRepository;
	protected Translator $translator;
	protected const QUANTITY_INPUT_PREFIX = 'eshopproductionWarehouseQuantity_';

	public function __construct(EntityManagerDecorator $em, WarehouseFacade $warehouseFacade,
		WarehouseRepository $warehouseRepository, Translator $translator)
	{
		$this->em = $em;
		$this->warehouseFacade = $warehouseFacade;
		$this->warehouseRepository = $warehouseRepository;
		$this->translator = $translator;
		$this->warehousePairs = $warehouseRepository->findPairs();
	}

	public static function getSubscribedEvents(): array
	{
		return [
			ProductForm::class . '::createForm'        => ['onCreateForm', 100],
			ProductForm::class . '::formSuccess'       => ['onFormSuccess', 100],
			ProductForm::class . '::beforeFormSuccess' => ['onBeforeFormSuccess', 100],
			ProductForm::class . '::formValidate'      => ['onFormValidate', 100],
			ProductForm::class . '::setProduct'        => ['onSetProduct', 100],
		];
	}

	public function onCreateForm(CreateFormEvent $event): void
	{
		$form = $event->form;
		/** @var BootstrapRenderer $renderer */
		$renderer = $form->getRenderer();

		foreach ($this->warehousePairs as $id => $name) {
			$inputName = self::QUANTITY_INPUT_PREFIX . $id;
			$form->addText($inputName, $this->translator->translate('eshopProductionWarehouse.productForm.eshopproductionWarehouseQuantity', null, ['warehouse' => $name]))
				 ->addCondition($form::FILLED)
				 	->addRule($form::FLOAT);
			$renderer->addToBaseExtendedLayout('right', $inputName);
		}
		$form->addCheckbox('isProductionRawMaterial', 'eshopProductionWarehouse.productForm.isProductionRawMaterial');
		$renderer->addToBaseExtendedLayout('right', 'isProductionRawMaterial');

		if (EshopProductionWarehouseConfig::load('enableNegativeQuantity', false)) {
			$form->addCheckbox('enableNegativeQuantity', 'eshopProductionWarehouse.productForm.enableNegativeQuantity');
			$renderer->addToBaseExtendedLayout('right', 'enableNegativeQuantity');
		}
	}

	public function onFormValidate(FormValidateEvent $event): void
	{
		if (!$event->custom['product'] && (Floats::isLessThan(self::getTotalQuantity($event->values), 0))) {
			$event->form->addError('eshopProductionWarehouse.productForm.errors.quantityGreaterThanZero');
		}
	}

	public function onBeforeFormSuccess(FormSuccessEvent $event): void
	{
		if (self::addProductMode($event->form)) {
			$event->values->quantity = (int) self::getTotalQuantity($event->values);
		}
	}

	public function onFormSuccess(FormSuccessEvent $event): void
	{
		/** @var Product|null $product */
		$product = $event->custom['entity'];
		$values = $event->values;
		$ef = $product ? $product->getExtraFieldsByKey() : [];

		$isProductionRawMaterial = $ef['isProductionRawMaterial'] ?? null;

		if (!$isProductionRawMaterial) {
			$isProductionRawMaterial = new ExtraField('isProductionRawMaterial', $values->isProductionRawMaterial);
			$product->fillExtraField($isProductionRawMaterial);
		}

		$isProductionRawMaterial->value = $values->isProductionRawMaterial;
		$this->em->persist($isProductionRawMaterial);

		if (EshopProductionWarehouseConfig::load('enableNegativeQuantity', false)) {
			$enableNegativeQuantity = $ef['enabledNegativeQuantity'] ?? null;

			if (!$enableNegativeQuantity) {
				$enableNegativeQuantity = new ExtraField('enabledNegativeQuantity', $values->enableNegativeQuantity);
				$product->fillExtraField($enableNegativeQuantity);
			}

			$enableNegativeQuantity->value = $values->enableNegativeQuantity;
			$this->em->persist($enableNegativeQuantity);
		}

		if (self::addProductMode($event->form)) {
			foreach ($event->values as $name => $value) {
				if (Strings::startsWith($name, self::QUANTITY_INPUT_PREFIX)) {
					$warehouse = $this->warehouseRepository->get((int) (explode('_', $name)[1]));

					$wm = new WarehouseMovement(new DateTime);
					$wm->warehouse = $warehouse;
					$wm->addInput(new WarehouseMovementInput($product, $wm, (float) $value));
					$this->em->persist($wm);
				}
			}
		}
	}

	public function onSetProduct(SetFormDataEvent $event): void
	{
		/** @var Product $entity */
		$entity = $event->entity;
		$ef = $entity->getExtraFieldsByKey();

		$event->form->getComponent('isProductionRawMaterial')->setDefaultValue($ef['isProductionRawMaterial']->value);

		$totalQuantity = null;
		foreach (array_keys($this->warehousePairs) as $warehouseId) {
			$form = $event->form;
			$products = $this->warehouseFacade->getState(null, [$entity->getId()], $warehouseId);
			if (isset($products[$entity->getId()])) {
				$quantity = $products[$entity->getId()]['quantity'];
				$totalQuantity = $totalQuantity ?? 0;
				$totalQuantity += $quantity;
				$form->getComponent(self::QUANTITY_INPUT_PREFIX . $warehouseId)->setDisabled()->setDefaultValue($quantity);
			} else {
				$form->getComponent(self::QUANTITY_INPUT_PREFIX . $warehouseId)->setDisabled()->setDefaultValue(0);
			}
		}

		if ($totalQuantity !== null) {
			$event->form->getComponent('quantity')->setDefaultValue($totalQuantity);
		}

		if (EshopProductionWarehouseConfig::load('enableNegativeQuantity', false)) {
			$event->form->getComponent('enableNegativeQuantity')->setDefaultValue($ef['enabledNegativeQuantity']->value);
		}
	}

	protected static function getTotalQuantity(ArrayHash $values): float
	{
		$quantityTotal = 0.0;
		foreach ($values as $name => $value) {
			if (Strings::startsWith($name, self::QUANTITY_INPUT_PREFIX)) {
				$quantityTotal += (float) $value;
			}
		}

		return $quantityTotal;
	}

	protected static function addProductMode(BaseForm $form): bool
	{
		/** @var ProductForm $form */
		$form = $form->parent;
		return !$form->presenter->getParameter('id');
	}

}