<?php declare(strict_types = 1);

namespace EshopProductionWarehouse\Model\Console;

use Core\Model\Entities\EntityManagerDecorator;
use Doctrine\ORM\Tools\Console\Command\SchemaTool\AbstractCommand;
use EshopProductionWarehouse\Model\Entities\Warehouse;
use Nette\Utils\DateTime;
use Nette\Utils\Floats;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Tracy\Debugger;
use Doctrine;

class InitState extends AbstractCommand
{
	/** @var EntityManagerDecorator @inject */
	public $em;

	protected function configure(): void
	{
		$this->setName('eshopproductionwarehouse:warehouse:initState');
	}

	protected function executeSchemaCommand(InputInterface $input, OutputInterface $output, Doctrine\ORM\Tools\SchemaTool $schemaTool, array $metadatas, SymfonyStyle $ui): int
	{
		$output->writeln('EshopProductionWarehouse INIT STATE');
		$connection = $this->em->getConnection();

		try {
			ini_set('memory_limit', '5G');
			Debugger::dispatch();

			$connection->transactional(function() use ($connection) {

				$connection->prepare("DELETE FROM eshop_production_warehouse__warehouse_movement")
						   ->execute();

				$connection->prepare("UPDATE eshop_production_warehouse__numerical_series SET counter = 1")
						   ->execute();

				$products = $connection->fetchAll("SELECT id, quantity FROM eshop_catalog__product");

				$eshopWarehouseId = $connection->fetchOne("SELECT id FROM eshop_production_warehouse__warehouse WHERE ident = ?", [Warehouse::IDENT_ESHOP_WAREHOUSE]);

				$connection->prepare("INSERT INTO eshop_production_warehouse__warehouse_movement(date, warehouse_id) VALUES (?, ?)")
						   ->execute([ new DateTime, $eshopWarehouseId]);

				$movementId = $connection->lastInsertId();

				foreach ($products as $product) {
					$id = $product['id'];
					$quantity = $product['quantity'];

					if (Floats::isLessThan($quantity, 0.0)) {
						$connection->prepare("INSERT INTO eshop_production_warehouse__warehouse_movement_output (id_product, id_warehouse_movement, quantity) VALUES (?, ?, ?)")
										 ->execute([
											 'id_product'            => $id,
											 'id_warehouse_movement' => $movementId,
											 'quantity'              => abs($quantity)
										 ]);
					} else if (Floats::isGreaterThan($quantity, 0.0)) {
						$connection->prepare("INSERT INTO eshop_production_warehouse__warehouse_movement_input (id_product, id_warehouse_movement, quantity) VALUES (?, ?, ?)")
										 ->execute([
											 'id_product'            => $id,
											 'id_warehouse_movement' => $movementId,
											 'quantity'              => abs($quantity)
										 ]);
					}
				}
			});

			$output->writeLn('<info>[OK]</info>');

			return 0;
		} catch (\Exception $e) {
			$output->writeln('<error>[ERROR] - ' . $e->getMessage() . '</error>');
		}

		return 1;
	}

}