<?php declare(strict_types = 1);

namespace EshopProductionWarehouse\Model\Entities;

use Core\Model\Entities\TId;
use DateTimeInterface;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use EshopCatalog\Model\Entities\Supplier;
use Doctrine\ORM\Mapping as ORM;

/**
 * @ORM\Entity
 * @ORM\Table(name="eshop_production_warehouse__warehouse_movement")
 * @ORM\EntityListeners({"EshopProductionWarehouse\Model\Listeners\WarehouseMovementListener"})
 */
class WarehouseMovement
{
	use TId;

	/**
	 * @ORM\Column(type="datetime", nullable=false)
	 */
	public DateTimeInterface $date;

	/**
	 * @ORM\ManyToOne(targetEntity="Warehouse")
	 * @ORM\JoinColumn(name="warehouse_id", referencedColumnName="id", nullable=true, onDelete="CASCADE")
	 */
	public ?Warehouse $warehouse = null;

	/**
	 * @var Collection<int, WarehouseMovementInput>
	 * @ORM\OneToMany(targetEntity="WarehouseMovementInput", mappedBy="warehouseMovement", cascade={"all"}, orphanRemoval=true)
	 */
	public Collection $inputs;

	/**
	 * @var Collection<int, WarehouseMovementOutput>
	 * @ORM\OneToMany(targetEntity="WarehouseMovementOutput", mappedBy="warehouseMovement", cascade={"all"}, orphanRemoval=true)
	 */
	public Collection $outputs;

	/**
	 * @ORM\ManyToOne(targetEntity="EshopCatalog\Model\Entities\Supplier")
	 * @ORM\JoinColumn(name="input_supplier_id", referencedColumnName="id", nullable=true, onDelete="CASCADE")
	 */
	public ?Supplier $inputSupplier = null;

	/**
	 * @ORM\ManyToOne(targetEntity="EshopCatalog\Model\Entities\Supplier")
	 * @ORM\JoinColumn(name="output_supplier_id", referencedColumnName="id", nullable=true, onDelete="CASCADE")
	 */
	public ?Supplier $outputSupplier = null;

	/**
	 * @ORM\Column(type="string", length=255, nullable=true)
	 */
	public ?string $inputInvoiceNumber = null;

	/**
	 * @ORM\Column(type="string", length=255, nullable=true)
	 */
	public ?string $outputInvoiceNumber = null;

	/**
	 * @ORM\Column(type="text", nullable=true)
	 */
	public ?string $inputDescription = null;

	/**
	 * @ORM\Column(type="text", nullable=true)
	 */
	public ?string $outputDescription = null;

	/**
	 * @ORM\Column(type="string", length=255, nullable=true)
	 */
	public ?string $inputNumber = null;

	/**
	 * @ORM\Column(type="string", length=255, nullable=true)
	 */
	public ?string $outputNumber = null;

	public function __construct(DateTimeInterface $date)
	{
		$this->date = $date;
		$this->inputs = new ArrayCollection;
		$this->outputs = new ArrayCollection;
	}

	public function addInput(WarehouseMovementInput $input): void
	{
		$this->inputs->add($input);
	}

	public function removeInput(WarehouseMovementInput $input): void
	{
		if ($this->inputs->contains($input)) {
			$this->inputs->removeElement($input);
		}
	}

	public function addOutput(WarehouseMovementOutput $output): void
	{
		$this->outputs->add($output);
	}

	public function removeOutput(WarehouseMovementOutput $output): void
	{
		if ($this->outputs->contains($output)) {
			$this->outputs->removeElement($output);
		}
	}

	/**
	 * @return WarehouseMovementInput[]
	 */
	public function getInputs(): array
	{
		return $this->inputs->toArray();
	}

	/**
	 * @return WarehouseMovementOutput[]
	 */
	public function getOutputs(): array
	{
		return $this->outputs->toArray();
	}

	/** @return int[] */
	public function getInputIds(): array
	{
		return array_map(fn(WarehouseMovementInput $input) => $input->getId(), $this->inputs->toArray());
	}

	/** @return int[] */
	public function getOutputIds(): array
	{
		return array_map(fn(WarehouseMovementOutput $output) => $output->getId(), $this->outputs->toArray());
	}

	public function getOutputById(int $id): ?WarehouseMovementOutput
	{
		foreach ($this->getOutputs() as $output) {
			if ($output->getId() === $id) {
				return $output;
			}
		}

		return null;
	}

	public function getInputById(int $id): ?WarehouseMovementInput
	{
		foreach ($this->getInputs() as $input) {
			if ($input->getId() === $id) {
				return $input;
			}
		}

		return null;
	}

}