<?php declare(strict_types = 1);

namespace EshopProductionWarehouse\Model\Subscribers;

use Core\Model\Entities\EntityManagerDecorator;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopProductionWarehouse\AdminModule\Model\Repository\WarehouseRepository;
use EshopProductionWarehouse\Model\Entities\Warehouse;
use EshopProductionWarehouse\Model\Entities\WarehouseMovement;
use EshopProductionWarehouse\Model\Entities\WarehouseMovementInput;
use EshopProductionWarehouse\Model\Entities\WarehouseMovementOutput;
use EshopProductionWarehouse\Model\Entities\WarehouseOrder;
use Nette\Utils\DateTime;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class OrderSubscriber implements EventSubscriberInterface
{
	protected EntityManagerDecorator $em;
	protected WarehouseRepository $warehouseRepository;

	public function __construct(EntityManagerDecorator $em, WarehouseRepository $warehouseRepository)
	{
		$this->em = $em;
		$this->warehouseRepository = $warehouseRepository;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.orderOnSuccess' => ['eshopOrder', 105],
			'eshopOrders.admin.beforeOrderChange' => ['beforeEshopCheckoutOrderChange', 105],
			'eshopOrders.admin.orderOnSuccess' => ['eshopCheckoutOrder', 105],
		];
	}

	/**
	 * Pouziva se pri platbe objednavky z eshopu na pokladne.
	 * Mnozstvi pred editaci objednavky se vraci zpet na sklad eshopu jako prijemka
	 */
	public function beforeEshopCheckoutOrderChange(OrderEvent $event): void
	{
		$order = $event->order;

		if ($order->getId() === null) {
			return;
		}

		PseudoWarehouseSubscriber::$enableAfterIncreaseProductsQuantity = false;
		PseudoWarehouseSubscriber::$enableAfterReduceProductsQuantity = false;

		/** @var WarehouseOrder|null $wo */
		$wo = $this->em->getRepository(WarehouseOrder::class)->findOneBy(['order' => $order->getId()]);
		$warehouse = $this->warehouseRepository->findByIdent(Warehouse::IDENT_ESHOP_WAREHOUSE);

		// overeni, ze objednavka pochazi opravdu se skladu eshopu
		if (!$warehouse || !$wo || $wo->warehouse->getId() !== $warehouse->getId()) {
			return;
		}

		$warehouseMovement = new WarehouseMovement(new DateTime);
		$warehouseMovement->warehouse = $warehouse;

		foreach ($order->getOrderItems() as $orderItem) {
			$product = $orderItem->getProduct();
			$quantity = (float) $orderItem->getQuantity();
			$warehouseMovementInput = new WarehouseMovementInput($product, $warehouseMovement, $quantity);
			$warehouseMovement->addInput($warehouseMovementInput);
		}

		$this->em->persist($warehouseMovement);

		$this->em->flush($warehouseMovement);
	}

	public function eshopCheckoutOrder(OrderEvent $event): void
	{
		$this->orderOnSuccess($event, Warehouse::IDENT_ESCHOPCHECKOUT_WAREHOUSE);
	}

	public function eshopOrder(OrderEvent $event): void
	{
		$this->orderOnSuccess($event, Warehouse::IDENT_ESHOP_WAREHOUSE);
	}

	protected function orderOnSuccess(OrderEvent $event, string $warehouseIdent): void
	{
		$order = $event->order;

		$warehouse = $this->warehouseRepository->findByIdent($warehouseIdent);

		if (!$warehouse) {
			return;
		}

		/**
		 * Pokud objednavka existuje na jinem skladu, tak ji prevedeme na spravny sklad
		 * @var WarehouseOrder|null $wo
		 */
		$wo = $this->em->getRepository(WarehouseOrder::class)->findOneBy(['order' => $order->getId()]);
		$warehouseOrder = $wo ?? new WarehouseOrder($warehouse, $order);
		$warehouseOrder->warehouse = $warehouse;
		$this->em->persist($warehouseOrder)->flush($warehouseOrder);

		/**
		 * Pouziva se pri platbe objednavky z eshopu na pokladne.
		 * Mnozstvi po editaci objednavky se odecte ze skladu pokladny jako vydejka
		 */
		if ($order->getId() !== null && $warehouseIdent === Warehouse::IDENT_ESCHOPCHECKOUT_WAREHOUSE) {
			/** @var WarehouseOrder|null $wo */
			$wo = $this->em->getRepository(WarehouseOrder::class)->findOneBy(['order' => $order->getId()]);

			if (!$wo) {
				return;
			}

			$warehouseMovement = new WarehouseMovement(new DateTime);
			$warehouseMovement->warehouse = $warehouse;

			foreach ($order->getOrderItems() as $orderItem) {
				$product = $orderItem->getProduct();
				$quantity = (float) $orderItem->getQuantity();
				$warehouseMovementOutput = new WarehouseMovementOutput($product, $warehouseMovement, $quantity);
				$warehouseMovement->addOutput($warehouseMovementOutput);
			}

			$this->em->persist($warehouseMovement);

			$this->em->flush($warehouseMovement);
		}
	}

}