<?php declare(strict_types = 1);

namespace EshopProductionWarehouse\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Doctrine\Common\Collections\ArrayCollection;
use EshopProductionWarehouse\AdminModule\Model\Facade\UsersFacade;
use EshopProductionWarehouse\AdminModule\Model\Facade\WarehouseFacade;
use EshopProductionWarehouse\AdminModule\Model\Repository\WarehouseRepository;
use EshopProductionWarehouse\Model\Entities\Batch;
use EshopProductionWarehouse\Model\Entities\Warehouse;
use EshopProductionWarehouse\Model\Entities\WarehouseMovement;
use EshopProductionWarehouse\Model\Entities\WarehouseMovementInput;
use EshopProductionWarehouse\Model\Entities\WarehouseMovementOutput;
use EshopProductionWarehouse\Model\EshopProductionWarehouseConfig;
use EshopProductionWarehouse\Model\Utils\Validators;
use Exception;
use Nette\Utils\ArrayHash;
use Nette\Utils\DateTime;
use Tracy\Debugger;
use Users\Model\Entities\User;

class ProductionForm extends BaseControl
{
	protected WarehouseFacade $warehouseFacade;
	protected WarehouseRepository $warehouseRepository;
	protected UsersFacade $usersFacade;
	protected ?Warehouse $eshopWarehouse = null;

	public function __construct(WarehouseFacade $warehouseFacade, WarehouseRepository $warehouseRepository, UsersFacade $usersFacade)
	{
		$this->warehouseFacade = $warehouseFacade;
		$this->warehouseRepository = $warehouseRepository;
		$this->usersFacade = $usersFacade;
		$this->eshopWarehouse = $warehouseRepository->findByIdent(Warehouse::IDENT_ESHOP_WAREHOUSE);
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	public function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setShowLangSwitcher(false)
			 ->setAjax();

		$form->addDatePicker('date', 'eshopProductionWarehouse.productionForm.date')
			 ->setDefaultValue(new DateTime)
			 ->setRequired();
		$form->addTextArea('description', 'eshopProductionWarehouse.productionForm.description')
			 ->setNullable();
		$form->addSelect(
			'batch',
			'eshopProductionWarehouse.productionForm.batch',
			['' => ''] + $this->em->getRepository(Batch::class)->findPairs([], 'name', ['name' => 'asc'], 'id')
		)->setRequired();
		$form->addSelect('warehouse', 'eshopProductionWarehouse.productionForm.warehouse', [null => null] + $this->warehouseRepository->findPairs())->setRequired();
		$form->addCheckboxList('packers', 'eshopProductionWarehouse.productionForm.packers', $this->usersFacade->getUsersWithPackingPrivilegePairs());
		$form->addDatePicker('expirationDate', 'eshopProductionWarehouse.productionForm.expirationDate')
			 ->setDefaultValue((new DateTime)->modify('+1 year'));
		$form->addInteger('quantity', 'eshopProductionWarehouse.productionForm.quantity')
			 ->setDefaultValue(1)
			 ->setRequired();
		$form->addText('batchNumber', 'eshopProductionWarehouse.productionForm.batchNumber')
			 ->setNullable();

		$form->addSaveCancelControl()
			 ->closeModalOnCancel();

		unset($form['saveControl']['save']);

		$form->onValidate[] = [$this, 'formValidate'];
		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values): void
	{
		/** @var Batch|null $batch */
		$batch = $this->em->getRepository(Batch::class)->find((int) $values->batch);

		if ($batch) {
			$hasWarehouseEnableNegativeQuantity = EshopProductionWarehouseConfig::load('enableNegativeQuantity');
			$productBatchState = [];

			// vstupy dávky jsou úbytkem na sklade
			foreach ($batch->getInputs() as $batchInput) {
				$product = $batchInput->getProduct();
				$productBatchState[$product->getId()] = [
					'id'                      => $product->getId(),
					'name'                    => $product->getText()->name,
					'quantity'                => (-$batchInput->getQuantity()) * $values->quantity,
					'unlimitedQuantity'       => (bool) $product->unlimitedQuantity,
					'enabledNegativeQuantity' => (bool) $product->getExtraFieldsValues()['enabledNegativeQuantity']
				];
			}

			$warehouseState = $this->warehouseFacade->getState(null, array_keys($productBatchState), $this->eshopWarehouse ? $this->eshopWarehouse->getId() : null);

			// situce, kdy produkt je v davce, ale neprosel skladem (ani jedna prijemka ci vydejka) -> je ho nedostatecne mnozstvi (0)
			foreach ($productBatchState as $productId => $productBatch) {
				if ($productBatch['unlimitedQuantity']) {
					continue;
				}

				if (!isset($warehouseState[$productId]) && (!$hasWarehouseEnableNegativeQuantity || !$productBatch['enabledNegativeQuantity'])) {
					$form->addError($this->t('eshopProductionWarehouse.productionForm.errors.productQuantityNotEnough', null, ['product' => $productBatch['name']]), false);
				}
			}

			foreach ($warehouseState as $productId => $product) {

				// resime jen produkty, ktere jsou v davce a ktere nemají nastavene neomezene mnozstvi, ostatní se preskakuji
				if (!isset($productBatchState[$productId]) || $product['unlimitedQuantity']) {
					continue;
				}

				$batchQuantity = $productBatchState[$productId]['quantity'];
				if (Validators::isDisallowedProductQuantity($product, $batchQuantity)) {
					$form->addError($this->t('eshopProductionWarehouse.productionForm.errors.productQuantityNotEnough', null, ['product' => $product['name']]), false);
				}
			}
		}

		$this->redrawControl('formErrors');
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): void
	{
		try {
			$this->em->beginTransaction();

			/** @var Batch|null $batch */
			$batch = $this->em->getRepository(Batch::class)->find((int) $values->batch);

			if (!$batch) {
				return;
			}

			$now = new DateTime;
			$values['date'] = $values['date']->setTime((int) $now->format('H'), (int) $now->format('i'), (int) $now->format('s'));
			$expirationDate = $values->expirationDate ? ($values->expirationDate)->setTime(23, 59, 59) : null;
			$packers = array_filter(array_map(fn($id) => $this->em->getRepository(User::class)->find($id), $values->packers), static fn($val) => !\Nette\Utils\Validators::isNone($val));

			if ($batch->getInputs()) {
				$warehouseMovement = new WarehouseMovement($values['date']);
				$warehouseMovement->warehouse = $this->eshopWarehouse;
				$warehouseMovement->outputDescription = $values->description;
				$warehouseMovement->packers = new ArrayCollection($packers);
				$warehouseMovement->expirationDate = $expirationDate;
				$warehouseMovement->isProduction = true;
				$warehouseMovement->batchNumber = $values->batchNumber;

				// vstupy dávky jsou úbytkem na sklade
				foreach ($batch->getInputs() as $input) {
					$product = $input->getProduct();
					$quantity = $input->getQuantity() * $values->quantity;
					$warehouseMovementOutput = new WarehouseMovementOutput($product, $warehouseMovement, $quantity);
					$warehouseMovement->addOutput($warehouseMovementOutput);

					$product->quantity -= (int) $quantity;
					$this->em->persist($product);
				}

				$this->em->persist($warehouseMovement);
			}

			if ($batch->getOutputs()) {
				$warehouseMovement = new WarehouseMovement($values['date']);
				$warehouseMovement->warehouse = $this->warehouseRepository->get((int) $values->warehouse);
				$warehouseMovement->inputDescription = $values->description;
				$warehouseMovement->packers = new ArrayCollection($packers);
				$warehouseMovement->expirationDate = $expirationDate;
				$warehouseMovement->isProduction = true;
				$warehouseMovement->batchNumber = $values->batchNumber;

				// vystupy dávky jsou prirustkem na sklade
				foreach ($batch->getOutputs() as $output) {
					$product = $output->getProduct();
					$quantity = $output->getQuantity() * $values->quantity;
					$warehouseMovementInput = new WarehouseMovementInput($product, $warehouseMovement, $quantity);
					$warehouseMovement->addInput($warehouseMovementInput);

					$product->quantity += (int) $quantity;
					$this->em->persist($product);
				}

				$this->em->persist($warehouseMovement);
			}

			$this->em->flush();
			$this->em->commit();

		} catch (Exception $exception) {
			Debugger::log($exception);
			$this->em->rollback();
			$form->addError($exception->getMessage());
			$this->redrawControl('formErrors');
		}
	}

}