<?php declare(strict_types = 1);

namespace EshopProductionWarehouse\AdminModule\Components;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\DataGrid\DataSource\DoctrineDataSource;
use DateTimeInterface;
use Doctrine\Common\Collections\Criteria;
use Doctrine\ORM\QueryBuilder;
use EshopProductionWarehouse\AdminModule\Model\Facade\UsersFacade;
use EshopProductionWarehouse\Model\Entities\WarehouseMovement;
use EshopProductionWarehouse\Model\Entities\WarehouseMovementInput;
use EshopProductionWarehouse\Model\Entities\WarehouseMovementOutput;
use Nette\Application\UI\Multiplier;
use Nette\Utils\DateTime;
use Users\Model\Entities\User;

class WarehouseMovementsGrid extends BaseControl
{
	protected QueryBuilder $queryBuilder;
	protected IWarehouseMovementsDetailFactory $movementsDetailFactory;
	protected UsersFacade $usersFacade;
	protected array $packersForDatagrid = [];

	public function __construct(IWarehouseMovementsDetailFactory $movementsDetailFactory, EntityManagerDecorator $em, UsersFacade $usersFacade)
	{
		$qb = $em->getRepository(WarehouseMovement::class)->createQueryBuilder('w');
		$qb->addOrderBy('w.date', 'desc');

		$this->queryBuilder = $qb;
		$this->movementsDetailFactory = $movementsDetailFactory;
		$this->usersFacade = $usersFacade;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	public function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$dataSource = new DoctrineDataSource($this->queryBuilder, 'w.id');

		$packers = $this->usersFacade->getUsersWithPackingPrivilegePairs();
		$dataSource->onDataLoaded[] = function(array $items) use ($packers) {
			/** @var WarehouseMovement[] $items */
			$ids = array_map(static fn($row) => $row->getId(), $items);

			if ($ids) {
				$p = $this->em->getConnection()->fetchAllAssociative("SELECT warehouse_movement_id, user_id FROM eshop_production_warehouse__warehouse_packer WHERE warehouse_movement_id IN (" . implode(',', $ids) . ")");

				foreach ($p as $row) {
					if (!isset($packers[$row['user_id']])) {
						continue;
					}

					if (!isset($this->packersForDatagrid[$row['warehouse_movement_id']])) {
						$this->packersForDatagrid[$row['warehouse_movement_id']] = $packers[$row['user_id']];
					} else {
						$this->packersForDatagrid[$row['warehouse_movement_id']] .= ', ' . $packers[$row['user_id']];
					}
				}
			}
		};

		$grid->setDataSource($dataSource);

		$grid->addColumnDateTime('date', 'eshopProductionWarehouse.warehouseMovementsGrid.date')
			 ->setAlign('left')
			 ->setFormat('d.m.Y')
			 ->setFilterDate();
		$grid->addColumnText('inputNumber', 'eshopProductionWarehouse.warehouseMovementsGrid.inputNumber')
			 ->setFilterText();
		$grid->addColumnText('outputNumber', 'eshopProductionWarehouse.warehouseMovementsGrid.outputNumber')
			 ->setFilterText();
		$grid->addColumnText('inputDescription', 'eshopProductionWarehouse.warehouseMovementsGrid.inputDescription')
			 ->setFilterText();
		$grid->addColumnText('outputDescription', 'eshopProductionWarehouse.warehouseMovementsGrid.outputDescription')
			 ->setFilterText();
		$grid->addColumnText('packers', 'eshopProductionWarehouse.warehouseMovementsGrid.packers')
			 ->setRenderer(function(WarehouseMovement $movement) {
				 return $this->packersForDatagrid[$movement->getId()] ?? '';
			 })
			 ->setFitContent()
			 ->setFilterText()->setCondition([$this, 'conditionGridPackers']);
		$grid->addColumnText('batchNumber', 'eshopProductionWarehouse.warehouseMovementsGrid.batchNumber')
			 ->setFitContent()
			 ->setFilterText();
		$grid->addColumnDateTime('expirationDate', 'eshopProductionWarehouse.warehouseMovementsGrid.expirationDate')
			 ->setFitContent()
			 ->setFormat('j. n. Y')
			 ->setAlign('left');
		$grid->addFilterDateRange('expirationDate', 'eshopProductionWarehouse.warehouseMovementsGrid.expirationDate')
			 ->setCondition(function(\Core\Model\Entities\QueryBuilder $qb, $values) {
				 if ($values->from) {
					 $date = DateTime::createFromFormat('j. n. Y', $values->from);
					 if ($date instanceof DateTimeInterface) {
						 $qb->andWhere('w.expirationDate >= :expirationDateFrom')
							->setParameter('expirationDateFrom', $date->setTime(23, 59, 59));
					 }
				 }

				 if ($values->to) {
					 $date = DateTime::createFromFormat('j. n. Y', $values->to);
					 if ($date instanceof DateTimeInterface) {
						 $qb->andWhere('w.expirationDate <= :expirationDateTo')
							->setParameter('expirationDateTo', $date->setTime(23, 59, 59));
					 }
				 }
			 })->setFormat('j. n. Y', 'j. n. Y');
		$grid->addColumnText('isProduction', 'eshopProductionWarehouse.warehouseMovementsGrid.isProduction')
			 ->setFitContent()
			 ->setRenderer(fn(WarehouseMovement $movement) => $this->t('default.' . ($movement->isProduction ? 'yes' : 'no')))
			 ->setFilterSelect([
				 ''  => '',
				 '1' => $this->t('default.yes'),
				 '0' => $this->t('default.no')
			 ]);
		$grid->setItemsDetail(__DIR__ . '/WarehouseMovementsGridDetail.latte');
		$grid->addAction('editInputs', 'eshopProductionWarehouse.warehouseMovementsGrid.editInputs', 'editInputs')
			 ->setRenderCondition(fn(WarehouseMovement $movement) => count($movement->getInputs()) > 0)
			 ->setIcon('edit')
			 ->setBsType('primary');
		$grid->addAction('editOutputs', 'eshopProductionWarehouse.warehouseMovementsGrid.editOutputs', 'editOutputs')
			 ->setRenderCondition(fn(WarehouseMovement $movement) => count($movement->getOutputs()) > 0)
			 ->setIcon('edit')
			 ->setBsType('primary');
		$grid->addAction('addInputsFromOutputs', 'eshopProductionWarehouse.warehouseMovementsGrid.addInputsFromOutputs', 'addInputsFromOutputs')
			 ->setRenderCondition(fn(WarehouseMovement $movement) => count($movement->getOutputs()) > 0)
			 ->setIcon('plus')
			 ->setBsType('success');
		$grid->addAction('addOutputsFromInputs', 'eshopProductionWarehouse.warehouseMovementsGrid.addOutputsFromInputs', 'addOutputsFromInputs')
			 ->setRenderCondition(fn(WarehouseMovement $movement) => count($movement->getInputs()) > 0)
			 ->setIcon('plus')
			 ->setBsType('success');

		$grid->addGroupAction('');

		$grid->addFilterText('products', 'eshopProductionWarehouse.warehouseMovementsGrid.searchInProducts')
			 ->setPlaceholder('eshopProductionWarehouse.warehouseMovementsGrid.code1')
			 ->setCondition(function(\Core\Model\Entities\QueryBuilder $qb, $value) {
				 $expr = $this->em->getExpressionBuilder();
				 $subInputQb = $this->em->getRepository(WarehouseMovementInput::class)->createQueryBuilder('wmi')
										->select('IDENTITY(wmi.warehouseMovement)')
										->join('wmi.product', 'p')
										->andWhere('p.code1 LIKE :code');

				 $subOutputQb = $this->em->getRepository(WarehouseMovementOutput::class)->createQueryBuilder('wmo')
										->select('IDENTITY(wmo.warehouseMovement)')
										->join('wmo.product', 'pr')
										->andWhere('pr.code1 LIKE :code');

				 $qb->andWhere($expr->in('w.id', $subInputQb->getDQL()) . ' OR ' . $expr->in('w.id', $subOutputQb->getDQL()))
					->setParameter('code', "%{$value}%");
			 });

		$grid->setHeadFilters(['products']);

		return $grid;
	}

	public function createComponentWarehouseMovementDetail(): Multiplier
	{
		return new Multiplier(function($warehouseMovementId) {
			$qb = $this->queryBuilder;
			$qb->andWhere('w.id = :id')->setParameter('id', $warehouseMovementId);

			return $this->movementsDetailFactory->create($qb->getQuery()->getOneOrNullResult());
		});
	}

	public function conditionGridPackers(\Core\Model\Entities\QueryBuilder $qb, string $value): void
	{
		$criteria = Criteria::create();

		$qb->leftJoin('w.packers', 'u');

		if ($value == '-') {
			$criteria->orWhere(Criteria::expr()->isNull('packers'));
		} else {
			foreach (explode(',', $value) as $val) {
				$criteria->orWhere(Criteria::expr()->contains('u.lastname', trim($val)));
			}
		}

		$qb->addCriteria($criteria);
	}

}