<?php declare(strict_types = 1);

namespace EshopProductionWarehouse\AdminModule\Model\Facade;

use Contributte\Translation\Translator;
use Core\Model\Entities\EntityManagerDecorator;
use DateTimeInterface;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\Model\Entities\Product;
use EshopProductionWarehouse\AdminModule\Model\Common\IOperation;
use EshopProductionWarehouse\AdminModule\Model\ProductExtraFields;
use EshopProductionWarehouse\AdminModule\Model\Repository\WarehouseRepository;
use EshopProductionWarehouse\Model\Entities\Warehouse;
use EshopProductionWarehouse\Model\Entities\WarehouseMovement;
use EshopProductionWarehouse\Model\Entities\WarehouseMovementInput;
use EshopProductionWarehouse\Model\Entities\WarehouseMovementOutput;
use Nette\Utils\DateTime;

class WarehouseFacade
{
	protected EntityManagerDecorator $em;
	protected ProductExtraFields $productExtraFields;
	protected Translator $translator;
	protected WarehouseRepository $warehouseRepository;

	public function __construct(EntityManagerDecorator $em, ProductExtraFields $productExtraFields, Translator $translator, WarehouseRepository $warehouseRepository)
	{
		$this->em = $em;
		$this->productExtraFields = $productExtraFields;
		$this->translator = $translator;
		$this->warehouseRepository = $warehouseRepository;
	}

	/**
	 * @return array<array{id: int, name: string, quantity: float, unlimitedQuantity: bool, enabledNegativeQuantity: bool, units: string}>
	 */
	public function getState(?DateTime $toDate = null, ?array $productIds = null, ?int $warehouseId = null): array
	{
		$fnGetSumMovements = function(string $movement) use ($toDate, $productIds, $warehouseId) {
			$qb = $this->em->getRepository(WarehouseMovement::class)->createQueryBuilder('w');
			$qb->select('p.id, mpt.name, p.code1, p.units, SUM(m.quantity) quantity, p.unlimitedQuantity')
			   ->join(sprintf('w.%s', $movement), 'm')
			   ->join('m.product', 'p')
			   ->join('p.productTexts', 'mpt', Join::WITH, 'mpt.lang = :lang')
			   ->groupBy('p')
			   ->setParameters([
				   'lang' => $this->translator->getLocale()
			   ]);

			if ($warehouseId) {
				$qb->andWhere($qb->expr()->eq('w.warehouse', $warehouseId));
			}

			if ($toDate) {
				$qb->andWhere('w.date <= :date')
				   ->setParameter('date', $toDate->setTime(23, 59, 59));
			}

			if ($productIds) {
				$qb->andWhere($qb->expr()->in('p.id', $productIds));
			}

			$result = $qb->getQuery()->getArrayResult();
			$enabledNegativeQuantities = [];
			if (count($result)) {
				$enabledNegativeQuantityQb = $this->productExtraFields->getEnabledNegativeQuantityQb(array_map(static fn(array $data) => $data['id'], $result));
				$enabledNegativeQuantityQb->select('ef2.sectionKey, ef2.value')->indexBy('ef2', 'ef2.sectionKey')
										  ->setParameter('lang', $this->translator->getLocale())
										  ->setParameter('sn', Product::EXTRA_FIELD_SECTION)
										  ->setParameter('key2', 'enabledNegativeQuantity');
				$enabledNegativeQuantities = $enabledNegativeQuantityQb->getQuery()->getArrayResult();
			}
			foreach ($result as $key => $r) {
				if (isset($enabledNegativeQuantities[$r['id']])) {
					$result[$key]['enabledNegativeQuantity'] = $enabledNegativeQuantities[$r['id']]['value'];
				} else {
					$result[$key]['enabledNegativeQuantity'] = null;
				}
			}

			return $result;
		};

		$inputs = array_map(static function(array $data) {
			$data['quantity'] = (float) $data['quantity'];
			return $data;
		}, $fnGetSumMovements('inputs'));
		$outputs = array_map(static function(array $data) {
			$data['quantity'] = ((float) $data['quantity']) * -1;
			return $data;
		}, $fnGetSumMovements('outputs'));

		$items = [];
		foreach ([$inputs, $outputs] as $arr) {
			foreach ($arr as $item) {
				$items[] = $item;
			}
		}

		$result = [];
		/** @var array{id: int, name: string, quantity: float, unlimitedQuantity: int, enabledNegativeQuantity: bool|null, units: string} $item */
		foreach ($items as $item) {
			$item['unlimitedQuantity'] = (bool) $item['unlimitedQuantity'];
			$item['enabledNegativeQuantity'] = (bool) $item['enabledNegativeQuantity'];
			if (!isset($result[$item['id']])) {
				$result[$item['id']] = $item;
			} else {
				$result[$item['id']]['quantity'] += $item['quantity'];
			}
		}

		return $result;
	}

	public function getAveragePurchasePrice(int $productId): float
	{
		$avg = 0;
		$qb = $this->em->getRepository(WarehouseMovementInput::class)->createQueryBuilder('i');
		$qb->select('i.price')
		   ->where('i.price IS NOT NULL AND i.product = :productId')
		   ->setParameter('productId', $productId);

		$items = array_map(static fn($item) => (float) $item['price'], $qb->getQuery()->getArrayResult());

		if ($items) {
			$sum = array_sum($items);
			$count = count($items);

			return $sum / $count;
		}

		return $avg;
	}

	public function getWarehouseByOrder(int $orderId): ?Warehouse
	{
		$qb = $this->em->getRepository(Warehouse::class)->createQueryBuilder('w');
		$qb->join('w.orders', 'wo')
		   ->join('wo.order', 'o')
		   ->where('o.id = :oId')
		   ->setParameter('oId', $orderId)
		   ->setMaxResults(1);

		return $qb->getQuery()->getOneOrNullResult();
	}

	/**
	 * @param IOperation[] $products
	 */
	public function createTransfer(int $fromWarehouse, int $toWarehouse, DateTimeInterface $date, array $products, ?int $orderId = null): void
	{
		$warehouseMovement = new WarehouseMovement($date);
		$warehouseMovement->warehouse = $this->warehouseRepository->get($fromWarehouse);

		if ($orderId) {
			$warehouseMovement->outputDescription = $this->translator->translate('eshopProductionWarehouse.common.order', ['order' => $orderId]);
		}

		foreach ($products as $output) {
			$product = $output->getProduct();
			$quantity = $output->getQuantity();
			$warehouseMovementOutput = new WarehouseMovementOutput($product, $warehouseMovement, $quantity);
			$warehouseMovement->addOutput($warehouseMovementOutput);
		}

		$this->em->persist($warehouseMovement);

		$warehouseMovement = new WarehouseMovement($date);
		$warehouseMovement->warehouse = $this->warehouseRepository->get($toWarehouse);

		if ($orderId) {
			$warehouseMovement->inputDescription = $this->translator->translate('eshopProductionWarehouse.common.order', ['order' => $orderId]);
		}

		foreach ($products as $input) {
			$product = $input->getProduct();
			$quantity = $input->getQuantity();
			$warehouseMovementInput = new WarehouseMovementInput($product, $warehouseMovement, $quantity);
			$warehouseMovement->addInput($warehouseMovementInput);
		}

		$this->em->persist($warehouseMovement);
	}

}