<?php declare(strict_types = 1);

namespace EshopProductionWarehouse\AdminModule\Model\Subscribers;

use Core\Model\Event\ControlEvent;
use Core\Model\Event\Event;
use Core\Model\UI\DataGrid\BaseDataGrid;
use EshopCatalog\AdminModule\Components\Products\ProductsGrid;
use EshopCatalog\Model\Entities\Product;
use EshopProductionWarehouse\AdminModule\Model\Facade\WarehouseFacade;
use EshopProductionWarehouse\AdminModule\Model\Repository\WarehouseRepository;
use EshopProductionWarehouse\Model\EshopProductionWarehouseConfig;
use Nette\Utils\Html;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class ProductsGridSubscriber implements EventSubscriberInterface
{
	protected WarehouseRepository $warehouseRepository;
	protected WarehouseFacade $warehouseFacade;
	protected array $warehousePairs = [];
	protected array $products = [];
	protected array $productsByWarehouses = [];

	public function __construct(WarehouseFacade $warehouseFacade, WarehouseRepository $warehouseRepository)
	{
		$this->warehouseFacade = $warehouseFacade;
		$this->warehouseRepository = $warehouseRepository;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			ProductsGrid::class . '::beforeColumnQuantityRender' => 'beforeColumnQuantityRender',
			ProductsGrid::class . '::onDataLoaded'               => 'onDataLoaded',
			ProductsGrid::class . '::onAttach'                   => 'productsGridAttached'
		];
	}

	public function onDataLoaded(Event $event): void
	{
		$productIds = $event->data['ids'];

		if (EshopProductionWarehouseConfig::load('productsGrid.quantityByWarehouses')) {
			foreach ($this->warehousePairs as $id => $warehouse) {
				$this->productsByWarehouses[$id] = $this->warehouseFacade->getState(null, $productIds, $id);
			}
		} else {
			$this->products = $this->warehouseFacade->getState(null, $productIds);
		}
	}

	public function beforeColumnQuantityRender(Event $event): void
	{
		/** @var Product $product */
		$product = $event->data['product'];
		if (isset($this->products[$product->getId()])) {
			$event->data['renderedQuantity'] = $this->products[$product->getId()]['quantity'];
		}
	}

	public function productsGridAttached(ControlEvent $event): void
	{
		if (!EshopProductionWarehouseConfig::load('productsGrid.quantityByWarehouses')) {
			return;
		}

		$this->warehousePairs = $this->warehouseRepository->findPairs();

		/** @var ProductsGrid $control */
		$control = $event->control;

		/** @var BaseDataGrid $grid */
		$grid = $control['grid'];

		foreach ($this->warehousePairs as $id => $warehouse) {
			$grid->addColumnText('warehouse-' . $id, $warehouse)
				 ->setRenderer(function(Product $product) use ($id) {
					 if ($product->unlimitedQuantity) {
						 return '∞';
					 }

					 $p = $this->productsByWarehouses[$id][$product->getId()];

					 $quantity = !$p ? 0 : $p['quantity'];
					 $quantity ??= 0;
					 $html = Html::el('');
					 $html->addHtml(Html::el('span', [
						 'class' => 'btn btn-xs ' . ($quantity > 0 ? 'btn-success' : 'btn-danger'),
					 ])->setText($quantity));

					 return $html;
				 })->setFitContent();
		}

		$grid->removeColumn('quantity');
	}

}