<?php declare(strict_types = 1);

namespace EshopProductionWarehouse\Model\Console;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Helpers\Strings;
use Doctrine\ORM\Tools\Console\Command\SchemaTool\AbstractCommand;
use Doctrine\ORM\Tools\SchemaTool;
use EshopCatalog\Model\Entities\Product;
use EshopProductionWarehouse\AdminModule\Model\Facade\WarehouseFacade;
use EshopProductionWarehouse\AdminModule\Model\Repository\WarehouseRepository;
use EshopProductionWarehouse\Model\Entities\Warehouse;
use EshopProductionWarehouse\Model\Entities\WarehouseMovement;
use EshopProductionWarehouse\Model\Entities\WarehouseMovementInput;
use EshopProductionWarehouse\Model\EshopProductionWarehouseConfig;
use Nette\Utils\DateTime;
use Nette\Utils\Validators;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

class TakeInventory extends AbstractCommand
{
	/** @var EntityManagerDecorator @inject */
	public $em;

	/** @var WarehouseRepository @inject */
	public $warehouseRepository;

	/** @var WarehouseFacade @inject */
	public $warehouseFacade;

	protected function configure(): void
	{
		$this->setName('eshopproductionwarehouse:warehouse:takeInventory');
	}

	/**
	 * @inheritDoc
	 */
	protected function executeSchemaCommand(InputInterface $input, OutputInterface $output, SchemaTool $schemaTool, array $metadatas, SymfonyStyle $ui): ?int
	{
		$file = EshopProductionWarehouseConfig::load('inventory.file');
		$date = EshopProductionWarehouseConfig::load('inventory.date');

		if (!$date) {
			$output->writeln('<error>[ERROR] - Date not filled</error>');
			return 0;
		}

		$date = new DateTime($date);

		if (!file_exists($file)) {
			$output->writeln('<error>[ERROR] - File ' . $file . ' not found</error>');
			return 0;
		}

		$data = [];
		$fr = fopen($file, 'r');

		if ($fr === false) {
			$output->writeln('<error>[ERROR] - File ' . $file . ' can not read</error>');
			return 0;
		}

		$separator = ';';
		while (!feof($fr) && $line = fgetcsv($fr, null, $separator)) {
			$data[] = array_map(static fn($lineV) => Strings::toUtf8($lineV), $line);
		}
		fclose($fr);

		// priprava a ocisteni dat
		$i = 0;
		$result = [];
		$header = [];
		$productIds = [];
		foreach ($data as $k => $v) {
			if ($i === 0) {
				$header = $v;
			} else {
				foreach ($v as $key => $val) {
					unset($v[$key]);
					if (Validators::isNumeric($val)) {
						$v[$header[$key]] = max((float) $val, 0);
					} else if (Validators::isNone($val)) {
						$v[$header[$key]] = null;
					} else if ($val === 'x') {
						$v[$header[$key]] = 999999;
					} else {
						$v[$header[$key]] = $val;
					}

					if ($header[$key] === 'id') {
						$productIds[] = $v[$header[$key]];
					}
				}
				$result[] = $v;
			}
			$i++;
		}

		if (!$productIds) {
			$output->writeln('<error>[ERROR] - Zero products</error>');
			return 0;
		}

		$connection = $this->em->getConnection();
		$connection->transactional(function() use ($connection, $date, $result, $productIds) {

			$connection->prepare("DELETE FROM eshop_production_warehouse__warehouse_movement WHERE `date` <= ?")
					   ->execute([$date->format('Y-m-d H:i:s')]);

			$qb = $this->em->getRepository(Product::class)->createQueryBuilder('p', 'p.id');
			$qb->where($qb->expr()->in('p.id', $productIds));

			/** @var Product[] $products */
			$products = $qb->getQuery()->getResult();

			$eshopWarehouse = $this->warehouseRepository->findByIdent(Warehouse::IDENT_ESHOP_WAREHOUSE);
			$eshopCheckoutWarehouse = $this->warehouseRepository->findByIdent(Warehouse::IDENT_ESCHOPCHECKOUT_WAREHOUSE);

			/** @var Warehouse $warehouse */
			foreach ([$eshopWarehouse, $eshopCheckoutWarehouse] as $warehouse) {
				$movement = new WarehouseMovement($date);
				$movement->warehouse = $warehouse;

				foreach ($result as $row) {
					if (!isset($products[$row['id']])) {
						continue;
					}
					$movement->addInput(new WarehouseMovementInput($products[$row['id']], $movement, $row[$warehouse->ident]));
				}
				$this->em->persist($movement);
			}

			$this->em->flush();

			$state = $this->warehouseFacade->getState(null, $productIds);

			foreach ($products as $product) {
				if (isset($state[$product->getId()])) {
					$q = $state[$product->getId()]['quantity'];
					$product->quantity = (int) $q;
					$this->em->persist($product);
				}
			}

			$this->em->flush();

		});

		return 1;
	}
}