<?php declare(strict_types = 1);

namespace EshopProductionWarehouse\Model\Subscribers;

use Contributte\Translation\Translator;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\Event;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Templating\Template;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Packages;
use EshopOrders\AdminModule\Components\Order\ExpeditionConfirm;
use EshopOrders\AdminModule\Components\Order\OrderForm;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderHistory;
use EshopOrders\Model\Entities\OrderItem;
use EshopProductionWarehouse\AdminModule\Model\Common\IOperation;
use EshopProductionWarehouse\AdminModule\Model\Facade\WarehouseFacade;
use EshopProductionWarehouse\AdminModule\Model\Repository\WarehouseRepository;
use EshopProductionWarehouse\Model\Entities\Warehouse;
use EshopProductionWarehouse\Model\Entities\WarehouseMovement;
use EshopProductionWarehouse\Model\Entities\WarehouseMovementInput;
use EshopProductionWarehouse\Model\Entities\WarehouseMovementOutput;
use EshopProductionWarehouse\Model\Entities\WarehouseOrder;
use Nette\Utils\DateTime;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class OrderSubscriber implements EventSubscriberInterface
{
	protected EntityManagerDecorator $em;
	protected WarehouseRepository $warehouseRepository;
	protected WarehouseFacade $warehouseFacade;
	protected Translator $translator;
	protected Packages $packages;

	public function __construct(EntityManagerDecorator $em, WarehouseRepository $warehouseRepository, WarehouseFacade $warehouseFacade, Translator $translator, Packages $packages)
	{
		$this->em = $em;
		$this->warehouseRepository = $warehouseRepository;
		$this->warehouseFacade = $warehouseFacade;
		$this->translator = $translator;
		$this->packages = $packages;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			OrderForm::class . '::createExpeditionForm'  => ['createExpeditionForm', 105],
			OrderForm::class . '::expeditionFormSuccess' => ['expeditionFormSuccess', 105],
			'eshopOrders.orderOnSuccess'                 => ['eshopOrder', 105],
			'eshopOrders.admin.beforeOrderChange'        => ['beforeEshopCheckoutOrderChange', 105],
			'eshopOrders.admin.orderOnSuccess'           => ['eshopCheckoutOrder', 105],
			ExpeditionConfirm::class . '::onRender'      => ['onRenderExpeditionConfirm', 105],
		];
	}

	public function onRenderExpeditionConfirm(Event $event): void
	{
		/** @var Template $template */
		$template = $event->data['template'];
		$template->includes = [__DIR__ . '/expeditionConfirmSubmits.latte'];
	}

	public function createExpeditionForm(CreateFormEvent $event): void
	{
		/** @var Order $order */
		$order = $event->data['order'];
		$form = $event->form;

		/** @var WarehouseOrder|null $wo */
		$wo = $this->em->getRepository(WarehouseOrder::class)->findOneBy(['order' => $order->getId()]);

		$options = [null => 'eshopProductionWarehouse.expeditionForm.nothing'];

		if ($wo) {
			$isEshopWarehouse = $wo->warehouse->ident === Warehouse::IDENT_ESHOP_WAREHOUSE;

			// jen pro osobni odbery a jen pro objednavky z eshopu
			if ($isEshopWarehouse && $order->getSpeditionIdent() === 'pickup') {
				$options['deductFromEshopCheckoutWarehouse'] = 'eshopProductionWarehouse.expeditionForm.deductFromEshopCheckoutWarehouse';
			}

			if ($isEshopWarehouse) {
				$options['transferToEshopCheckoutWarehouse'] = 'eshopProductionWarehouse.expeditionForm.transferToEshopCheckoutWarehouse';
			}

			$form->addSelect('warehouseActions', 'eshopProductionWarehouse.expeditionForm.actions', $options)
				 ->setDefaultValue(null);

			if($wo->lastAction) {
				$form->getComponent('warehouseActions')->setDescription($this->translator->translate('eshopProductionWarehouse.expeditionForm.lastAction') . ': '. $this->translator->translate('eshopProductionWarehouse.expeditionForm.' . $wo->lastAction));
			}
		}
	}

	public function expeditionFormSuccess(FormSuccessEvent $event): void
	{
		/** @var Order $order */
		$order = $event->custom['order'];
		$values = $event->values;

		// funkce pro pridani checkboxu pro prevod objednavky s osobnim odber ze skladu eshopu na prodejnu
		if (array_key_exists('warehouseActions', (array) $values) && $values->warehouseActions === 'deductFromEshopCheckoutWarehouse') {
			/**
			 * Pokud objednavka existuje na jinem skladu, tak ji prevedeme na spravny sklad
			 * @var WarehouseOrder|null $wo
			 */
			$wo = $this->em->getRepository(WarehouseOrder::class)->findOneBy(['order' => $order->getId()]);
			$warehouse = $this->warehouseRepository->findByIdent(Warehouse::IDENT_ESCHOPCHECKOUT_WAREHOUSE);

			if ($wo && $warehouse) {
				$fromWarehouseId = (int) $wo->warehouse->getId();
				$toWarehouseId = (int) $warehouse->getId();
				$wo->warehouse = $warehouse;
				$wo->lastAction = $values->warehouseActions;
				$this->em->persist($wo)->flush($wo);

				/** @var IOperation[] $products */
				$products = [];
				foreach ($this->getOrderItems($order) as $orderItem) {
					/** @var Product|null $product */
					$product = $this->em->getRepository(Product::class)->find((int) $orderItem->getProductId());

					if (!$product) {
						continue;
					}

					$operation = new class(null, $product, $orderItem->getQuantity()) implements IOperation {

						protected ?int $id;
						protected Product $product;
						protected float $quantity;
						protected array $extras = [];

						public function __construct(?int $id, Product $product, float $quantity)
						{
							$this->id = $id;
							$this->product = $product;
							$this->quantity = $quantity;
						}

						public function getQuantity(): float
						{
							return $this->quantity;
						}

						public function setQuantity($quantity): self
						{
							$this->quantity = $quantity;
							return $this;
						}

						public function getProduct(): Product
						{
							return $this->product;
						}

						public function setProduct(Product $product): self
						{
							$this->product = $product;
							return $this;
						}

						public function getId(): ?int
						{
							return $this->id;
						}

						public function getExtras(): array
						{
							return $this->extras;
						}

						public function setExtras(array $extras): self
						{
							$this->extras = $extras;
							return $this;
						}

						public function setExtra($key, $val): self
						{
							$this->extras[$key] = $val;
							return $this;
						}
					};

					$products[] = $operation;
				}

				// cíl je aby vznikla prijemka na eshop a vydejka z pokladny (objednavka pochazi z eshopu)
				$this->warehouseFacade->createTransfer($toWarehouseId, $fromWarehouseId, new DateTime, $products, $order->getId());

				if ($values->expeditionConfirmRulesOk) {
					$orderHistory = new OrderHistory($order, $this->translator->translate('eshopProductionWarehouse.expeditionFormStatuses.readyInCheckout'), null);
					$this->em->persist($orderHistory);
				}

				$this->em->flush();
			}
		}

		// vytvori se prijemka na prodejnu
		if (array_key_exists('warehouseActions', (array) $values) && $values->warehouseActions === 'transferToEshopCheckoutWarehouse') {
			/**
			 * @var WarehouseOrder|null $wo
			 */
			$wo = $this->em->getRepository(WarehouseOrder::class)->findOneBy(['order' => $order->getId()]);
			$warehouse = $this->warehouseRepository->findByIdent(Warehouse::IDENT_ESCHOPCHECKOUT_WAREHOUSE);

			if ($wo && $warehouse) {
				$wo->warehouse = $warehouse;
				$wo->lastAction = $values->warehouseActions;
				$wo->countInStatistics = 0; // tyto objednavky nezapocitavat ve statistikach
				$wo->transferredToEshopCheckoutWarehouse = 1; // oznaceni, ze byla objednavka prevezena na prodejnu
				$this->em->persist($wo)->flush($wo);

				$warehouseMovement = new WarehouseMovement(new DateTime);
				$warehouseMovement->warehouse = $warehouse;
				$warehouseMovement->inputDescription = $this->translator->translate('eshopProductionWarehouse.common.order', ['order' => $order->getId()]);

				foreach ($this->getOrderItems($order) as $orderItem) {
					$product = $orderItem->getProduct();

					if (!$product) {
						continue;
					}

					$quantity = (float) $orderItem->getQuantity();

					$product->quantity += (int) $quantity;
					$this->em->persist($product);

					$warehouseMovementInput = new WarehouseMovementInput($product, $warehouseMovement, $quantity);
					$warehouseMovement->addInput($warehouseMovementInput);
				}

				$this->em->persist($warehouseMovement);

				// pri prevedeni na prodejnu neni potreba generovat faktura (je to jakasi interni objednavka)
				$order->enableInvoiceGeneration = 0;
				$this->em->persist($order);

				$this->em->flush();
			}
		}

		if (array_key_exists('warehouseActions', (array) $values) && $values->expeditionConfirmRulesOk && !$values->warehouseActions) {
			$orderHistory = new OrderHistory($order, $this->translator->translate('eshopProductionWarehouse.expeditionFormStatuses.readyInProduction'), null);
			$this->em->persist($orderHistory);

			$this->em->flush();
		}
	}

	/**
	 * Pouziva se pri platbe objednavky z eshopu na pokladne.
	 * Mnozstvi pred editaci objednavky se vraci zpet na sklad eshopu jako prijemka
	 */
	public function beforeEshopCheckoutOrderChange(OrderEvent $event): void
	{
		$order = $event->order;

		if ($order->getId() === null) {
			return;
		}

		PseudoWarehouseSubscriber::$enableAfterIncreaseProductsQuantity = false;
		PseudoWarehouseSubscriber::$enableAfterReduceProductsQuantity = false;

		/** @var WarehouseOrder|null $wo */
		$wo = $this->em->getRepository(WarehouseOrder::class)->findOneBy(['order' => $order->getId()]);
		$warehouse = $this->warehouseRepository->findByIdent(Warehouse::IDENT_ESHOP_WAREHOUSE);

		// overeni, ze objednavka pochazi opravdu se skladu eshopu
		if (!$warehouse || !$wo || $wo->warehouse->getId() !== $warehouse->getId()) {
			return;
		}

		$warehouseMovement = new WarehouseMovement(new DateTime);
		$warehouseMovement->warehouse = $warehouse;
		$warehouseMovement->inputDescription = $this->translator->translate('eshopProductionWarehouse.common.order', ['order' => $order->getId()]);

		foreach ($this->getOrderItems($order) as $orderItem) {
			$product = $orderItem->getProduct();

			if (!$product) {
				continue;
			}

			$quantity = (float) $orderItem->getQuantity();
			$warehouseMovementInput = new WarehouseMovementInput($product, $warehouseMovement, $quantity);
			$warehouseMovement->addInput($warehouseMovementInput);
		}

		$this->em->persist($warehouseMovement);

		$this->em->flush($warehouseMovement);
	}

	public function eshopCheckoutOrder(OrderEvent $event): void
	{
		$order = $event->order;

		$this->orderOnSuccess($event, $order->getParam('isInternalOrder') ? Warehouse::IDENT_ESHOP_WAREHOUSE : Warehouse::IDENT_ESCHOPCHECKOUT_WAREHOUSE);
	}

	public function eshopOrder(OrderEvent $event): void
	{
		$this->orderOnSuccess($event, Warehouse::IDENT_ESHOP_WAREHOUSE);
	}

	protected function orderOnSuccess(OrderEvent $event, string $warehouseIdent): void
	{
		$order = $event->order;

		$warehouse = $this->warehouseRepository->findByIdent($warehouseIdent);

		if (!$warehouse) {
			return;
		}

		/**
		 * Pokud objednavka existuje na jinem skladu, tak ji prevedeme na spravny sklad
		 * @var WarehouseOrder|null $wo
		 */
		$wo = $this->em->getRepository(WarehouseOrder::class)->findOneBy(['order' => $order->getId()]);
		$originWarehouseIdent = $wo ? $wo->warehouse->ident : null;
		$warehouseOrder = $wo ?? new WarehouseOrder($warehouse, $order);
		$warehouseOrder->warehouse = $warehouse;
		$this->em->persist($warehouseOrder)->flush($warehouseOrder);

		/**
		 * Pouziva se pri platbe objednavky z eshopu na pokladne.
		 * Mnozstvi po editaci objednavky se odecte ze skladu pokladny jako vydejka
		 */
		if ($order->getId() !== null && $originWarehouseIdent === Warehouse::IDENT_ESHOP_WAREHOUSE && $warehouseIdent === Warehouse::IDENT_ESCHOPCHECKOUT_WAREHOUSE) {
			/** @var WarehouseOrder|null $wo */
			$wo = $this->em->getRepository(WarehouseOrder::class)->findOneBy(['order' => $order->getId()]);

			if (!$wo) {
				return;
			}

			$warehouseMovement = new WarehouseMovement(new DateTime);
			$warehouseMovement->warehouse = $warehouse;
			$warehouseMovement->outputDescription = $this->translator->translate('eshopProductionWarehouse.common.order', ['order' => $order->getId()]);

			foreach ($this->getOrderItems($order) as $orderItem) {
				$product = $orderItem->getProduct();

				if (!$product) {
					continue;
				}

				$quantity = (float) $orderItem->getQuantity();
				$warehouseMovementOutput = new WarehouseMovementOutput($product, $warehouseMovement, $quantity);
				$warehouseMovement->addOutput($warehouseMovementOutput);
			}

			$this->em->persist($warehouseMovement);

			$this->em->flush($warehouseMovement);
		}
	}

	/**
	 * @return OrderItem[]
	 */
	protected function getOrderItems(Order $order): array
	{
		$editablePackageIds = $this->packages->getEditablePackageIds();

		$result = [];
		foreach ($order->getOrderItems() as $orderItem) {
			// vypustime obalku u editovatelnych balicku
			if (!$orderItem->getParent() && $orderItem->getProductId() && in_array($orderItem->getProductId(), $editablePackageIds, true)) {
				continue;
			}

			$result[] = $orderItem;
		}

		return $result;
	}

}