<?php declare(strict_types = 1);

namespace EshopProductionWarehouse\Model\Subscribers;

use Contributte\Translation\Translator;
use Core\Model\Application\AppState;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\Event;
use EshopCatalog\Model\Entities\Product;
use EshopOrders\Model\Entities\Order;
use EshopProductionWarehouse\AdminModule\Model\Facade\WarehouseFacade;
use EshopProductionWarehouse\AdminModule\Model\Repository\WarehouseRepository;
use EshopProductionWarehouse\Model\Entities\Warehouse;
use EshopProductionWarehouse\Model\Entities\WarehouseMovement;
use EshopProductionWarehouse\Model\Entities\WarehouseMovementInput;
use EshopProductionWarehouse\Model\Entities\WarehouseMovementOutput;
use EshopProductionWarehouse\Model\Entities\WarehouseOrder;
use Nette\Utils\DateTime;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class PseudoWarehouseSubscriber implements EventSubscriberInterface
{
	protected EntityManagerDecorator $em;
	protected WarehouseFacade $warehouseFacade;
	protected WarehouseRepository $warehouseRepository;
	protected Translator $translator;
	public static bool $enableAfterReduceProductsQuantity = true;
	public static bool $enableAfterIncreaseProductsQuantity = true;

	public function __construct(EntityManagerDecorator $em, WarehouseFacade $warehouseFacade, Translator $translator, WarehouseRepository $warehouseRepository)
	{
		$this->em = $em;
		$this->warehouseFacade = $warehouseFacade;
		$this->translator = $translator;
		$this->warehouseRepository = $warehouseRepository;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'pseudoWarehouse.afterReduceProductsQuantity'   => ['afterReduceProductsQuantity', 100],
			'pseudoWarehouse.afterIncreaseProductsQuantity' => ['afterIncreaseProductsQuantity', 100],
		];
	}

	public function afterReduceProductsQuantity(Event $event): void
	{
		if (!self::$enableAfterReduceProductsQuantity) {
			return;
		}

		/** @var Order|null $order */
		$order = $event->data['order'];

		/** @var Order|null $fromOrder */
		$fromOrder = AppState::getState('correctiveTaxDocumentFromOrder');
		if ($fromOrder) {
			$order = $fromOrder;
		}

		$warehouse = $order ? $this->warehouseFacade->getWarehouseByOrder($order->getId()) : null;
		
		/** @var array<int, array{product: Product, quantityDiff: int}> $changes */
		$changes = $event->data['changes'];

		$warehouseMovement = new WarehouseMovement(new DateTime);
		$warehouseMovement->warehouse = $warehouse;

		if ($order->getId()) {
			$warehouseMovement->outputDescription = $this->translator->translate('eshopProductionWarehouse.common.order', ['order' => $order->getId()]);
		}

		foreach ($changes as $change) {
			$warehouseMovementOutput = new WarehouseMovementOutput($change['product'], $warehouseMovement, $change['quantityDiff']);
			$warehouseMovement->addOutput($warehouseMovementOutput);
		}

		if (count($changes) > 0) {
			$this->em->persist($warehouseMovement)->flush($warehouseMovement);
		}
	}

	public function afterIncreaseProductsQuantity(Event $event): void
	{
		if (!self::$enableAfterIncreaseProductsQuantity) {
			return;
		}

		/** @var Order|null $order */
		$order = $event->data['order'];

		$isOrderCanceled = AppState::getState('orderIsCanceled', false);

		/** @var Order|null $fromOrder */
		$fromOrder = AppState::getState('correctiveTaxDocumentFromOrder');
		if ($fromOrder) {
			$order = $fromOrder;
			$isOrderCanceled = true;
		}

		$warehouse = $order ? $this->warehouseFacade->getWarehouseByOrder($order->getId()) : null;

		/** @var array<int, array{product: Product, quantityDiff: int}> $changes */
		$changes = $event->data['changes'];

		// pokud je objednávka stornovana (nebo ODD) a je to objednavka prevezena na prodejnu, pak se vytvori prijemka na eshop a vydejka na prodejnu
		if ($isOrderCanceled) {
			/** @var WarehouseOrder|null $warehouseOrder */
			$warehouseOrder = $this->em->getRepository(WarehouseOrder::class)->findOneBy(['order' => $order->getId()]);
			if ($warehouseOrder && $warehouseOrder->transferredToEshopCheckoutWarehouse) {

				$warehouseInputMovement = new WarehouseMovement(new DateTime);
				$warehouseInputMovement->warehouse = $this->warehouseRepository->findByIdent(Warehouse::IDENT_ESHOP_WAREHOUSE);

				if ($order->getId()) {
					$warehouseInputMovement->inputDescription = $this->translator->translate('eshopProductionWarehouse.common.order', ['order' => $order->getId()]);
				}

				foreach ($changes as $change) {
					$warehouseInputMovement->addInput(new WarehouseMovementInput($change['product'], $warehouseInputMovement, $change['quantityDiff']));
				}

				$warehouseOutputMovement = new WarehouseMovement(new DateTime);
				$warehouseOutputMovement->warehouse = $this->warehouseRepository->findByIdent(Warehouse::IDENT_ESCHOPCHECKOUT_WAREHOUSE);

				if ($order->getId()) {
					$warehouseOutputMovement->outputDescription = $this->translator->translate('eshopProductionWarehouse.common.order', ['order' => $order->getId()]);
				}

				foreach ($changes as $change) {
					$warehouseOutputMovement->addOutput(new WarehouseMovementOutput($change['product'], $warehouseOutputMovement, $change['quantityDiff']));
				}

				if (count($changes) > 0) {
					$this->em->persist($warehouseInputMovement);
					$this->em->persist($warehouseOutputMovement);
					$this->em->flush($warehouseInputMovement);
					$this->em->flush($warehouseOutputMovement);
				}

				return;
			}
		}

		$warehouseMovement = new WarehouseMovement(new DateTime);
		$warehouseMovement->warehouse = $warehouse;

		if ($order->getId()) {
			$warehouseMovement->inputDescription = $this->translator->translate('eshopProductionWarehouse.common.order', ['order' => $order->getId()]);
		}

		foreach ($changes as $change) {
			$warehouseMovementInput = new WarehouseMovementInput($change['product'], $warehouseMovement, $change['quantityDiff']);
			$warehouseMovement->addInput($warehouseMovementInput);
		}

		if (count($changes) > 0) {
			$this->em->persist($warehouseMovement)->flush($warehouseMovement);
		}
	}

}