<?php declare(strict_types = 1);

namespace EshopProductionWarehouse\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\AdminModule\Model\Suppliers;
use EshopCatalog\Model\Entities\Supplier;
use EshopProductionWarehouse\AdminModule\Model\Facade\ProductsFacade;
use EshopProductionWarehouse\AdminModule\Model\Facade\WarehouseFacade;
use EshopProductionWarehouse\AdminModule\Model\FormContainers\ProductsContainer\ProductsContainer;
use EshopProductionWarehouse\AdminModule\Model\FormContainers\ProductsContainer\ProductsContainerConfig;
use EshopProductionWarehouse\AdminModule\Model\Repository\WarehouseRepository;
use EshopProductionWarehouse\Model\Entities\WarehouseMovement;
use EshopProductionWarehouse\Model\Entities\WarehouseMovementOperation;
use EshopProductionWarehouse\Model\Utils\UnitsHelper;
use Nette\Utils\ArrayHash;
use Nette\Utils\DateTime;
use Tracy\Debugger;

class WarehouseMovementForm extends BaseControl
{
	protected ProductsContainer $productsContainer;
	protected ProductsFacade $productsFacade;
	protected WarehouseFacade $warehouseFacade;
	protected WarehouseRepository $warehouseRepository;
	protected Suppliers $suppliers;
	protected ?WarehouseMovement $warehouseMovement = null;
	/** @var WarehouseMovementOperation[] */
	protected array $warehouseMovementOperations = [];
	protected array $productItems;
	protected ProductsContainerConfig $config;
	protected $productsTextPrefix;
	/** @var callable|null */
	protected $onValidate = null;
	/** @var callable|null */
	protected $onSuccess = null;
	/** @var callable|null */
	protected $onSetDefaults = null;

	public function __construct(string $productsTextPrefix, ProductsContainerConfig $config, ProductsContainer $productsContainer, ProductsFacade $productsFacade, WarehouseFacade $warehouseFacade,
		Suppliers $suppliers, WarehouseRepository $warehouseRepository)
	{
		$this->config = $config;
		$this->suppliers = $suppliers;
		$this->productsTextPrefix = $productsTextPrefix;
		$this->productsContainer = $productsContainer;
		$this->productsFacade = $productsFacade;
		$this->productItems = $this->productsFacade->getAll();
		$this->warehouseFacade = $warehouseFacade;
		$this->warehouseRepository = $warehouseRepository;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	public function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setShowLangSwitcher(false)
			 ->setAjax();

		$suppliers = $this->getSuppliersPairs();
		uasort($suppliers, static fn($a, $b) => $a <=> $b);

		$form->addDatePicker('date', 'eshopProductionWarehouse.warehouseMovementForm.date')
			 ->setDefaultValue(new DateTime)
			 ->setRequired();
		$form->addText('invoiceNumber', 'eshopProductionWarehouse.warehouseMovementForm.invoiceNumber')
			 ->setNullable();
		$form->addSelect('warehouse', 'eshopProductionWarehouse.warehouseMovementForm.warehouse', [null => null] + $this->warehouseRepository->findPairs())
			 ->setDisabled($this->warehouseMovement !== null)
			 ->setRequired();
		$form->addSelect('supplier', 'eshopProductionWarehouse.warehouseMovementForm.supplier.caption', $suppliers)
			 ->setPrompt('eshopProductionWarehouse.warehouseMovementForm.supplier.prompt')
			 ->setRequired(false);
		$form->addTextArea('description', 'eshopProductionWarehouse.warehouseMovementForm.description')
			 ->setNullable();

		$productItemPairs = [];
		foreach ($this->productItems as $item) {
			$productItemPairs[$item['id']] = $item['name'];
		}

		$productUnits = [];
		foreach ($this->productItems as $id => $product) {
			$productUnits[$id] = [
				'allowedUnits' => array_map(fn($v) => $this->t('eshopCatalog.productForm.units.items.' . $v), UnitsHelper::findRelatedUnits($product['units'])),
				'unit'         => $product['units'],
				'unitText'     => $this->t('eshopCatalog.productForm.units.items.' . $product['units']),
			];
		}

		if ($this->warehouseMovement !== null) {
			$this->productsContainer->createBulk($productItemPairs, $productUnits, $this->productsTextPrefix, $form, 'products', $this->warehouseMovementOperations, $this->config);
		} else {
			$this->productsContainer->createOne($productItemPairs, $productUnits, $this->productsTextPrefix, $form, 'products', $this->config);
		}

		$form->addSaveCancelControl();

		$form->onValidate[] = [$this, 'formValidate'];
		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formValidate(BaseForm $form, ArrayHash $values): void
	{
		$form['supplier']->cleanErrors();
		$this->productsContainer->validate($this->productsTextPrefix, $form, 'products');
		if ($this->onValidate) {
			($this->onValidate)($form, $values, $this->productsContainer->process($form, 'products'), $this->warehouseMovement, $this->warehouseFacade);
		}
		$this->redrawControl('formErrors');
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): void
	{
		try {
			$this->em->beginTransaction();

			if (!$this->warehouseMovement) {
				$this->warehouseMovement = new WarehouseMovement($values['date']);
				$this->warehouseMovement->warehouse = $this->warehouseRepository->get((int) $values['warehouse']);
			}

			$this->warehouseMovement->date = $values['date'];

			if ($this->onSuccess) {
				($this->onSuccess)($form, $this->warehouseFacade, $this->warehouseMovement, $this->productsContainer->process($form, 'products'));
			}

			if ($form->hasErrors()) {
				$this->redrawControl('formErrors');
				return;
			}

			$this->em->persist($this->warehouseMovement)
					 ->flush();

			$this->em->commit();

			$form->addCustomData('id', $this->warehouseMovement->getId());

		} catch (\Exception $exception) {
			Debugger::log($exception);
			$this->em->rollback();
			$form->addError($exception->getMessage());
			$this->redrawControl('formErrors');
		}
	}

	/**
	 * @param WarehouseMovementOperation[] $warehouseMovementOperations
	 */
	public function setWarehouseMovement(WarehouseMovement $movement, array $warehouseMovementOperations): void
	{
		$this->warehouseMovement = $movement;
		$this->warehouseMovementOperations = $warehouseMovementOperations;
		$this['form']->setDefaults([
			'date'      => $movement->date,
			'warehouse' => $movement->warehouse->getId()
		]);
		if ($this->onSetDefaults) {
			($this->onSetDefaults)($this['form'], $movement);
		}
	}

	/** @param callable $callback */
	public function setOnValidateCallback($callback): void
	{
		$this->onValidate = $callback;
	}

	/** @param callable $callback */
	public function setOnSuccessCallback($callback): void
	{
		$this->onSuccess = $callback;
	}

	/** @param callable $callback */
	public function setOnSetDefaults($callback): void
	{
		$this->onSetDefaults = $callback;
	}

	/**
	 * @return Supplier[]
	 */
	protected function getSuppliersAssoc(): array
	{
		$result = [];
		foreach ($this->suppliers->getAll() as $supplier) {
			if ((bool) $supplier->isPublished) {
				$result[$supplier->getId()] = $supplier;
			}
		}

		return $result;
	}

	/**
	 * @return string[]
	 */
	protected function getSuppliersPairs(): array
	{
		$result = [];
		foreach ($this->getSuppliersAssoc() as $key => $supplier) {
			$result[$key] = $supplier->name;
		}

		return $result;
	}

}