<?php declare(strict_types = 1);

namespace EshopProductionWarehouse\AdminModule\Model\Subscribers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Entities\QueryBuilder;
use Core\Model\Event\ControlEvent;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\DataGrid\DataSource\DoctrineDataSource;
use Doctrine\ORM\Query\Expr\Join;
use EshopOrders\AdminModule\Components\Order\OrdersGrid;
use EshopOrders\Model\Entities\Order;
use EshopProductionWarehouse\AdminModule\Model\Repository\WarehouseRepository;
use EshopProductionWarehouse\Model\Entities\WarehouseOrder;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class OrdersGridSubscriber implements EventSubscriberInterface
{
	protected WarehouseRepository $warehouseRepository;
	protected EntityManagerDecorator $em;

	public function __construct(WarehouseRepository $warehouseRepository, EntityManagerDecorator $entityManager)
	{
		$this->warehouseRepository = $warehouseRepository;
		$this->em = $entityManager;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			OrdersGrid::class . '::onAttach' => 'ordersGridAttached'
		];
	}

	public function ordersGridAttached(ControlEvent $event): void
	{
		/** @var OrdersGrid $grid */
		$control = $event->control;

		/** @var BaseDataGrid $grid */
		$grid = $control['grid'];

		$orderWarehousePairs = [];

		/** @var DoctrineDataSource $qb */
		$ds = $grid->getDataSource();
		$ds->updateQueryBuilder(function(QueryBuilder $qb) {
			$qb->leftJoin(WarehouseOrder::class, 'wo', Join::WITH, 'wo.order = o.id');
		});

		/** @var Order[] $items */
		$grid->getDataSource()->onDataLoaded[] = function(array $items) use (&$orderWarehousePairs) {
			$ids = array_map(static fn(Order $o) => $o->getId(), $items);

			if ($ids) {
				$qb = $this->em->getRepository(WarehouseOrder::class)->createQueryBuilder('wo');
				$qb->select('IDENTITY(wo.order) as orderId, w.name as warehouseName')
				   ->join('wo.warehouse', 'w')
				   ->where($qb->expr()->in('wo.order', $ids));

				foreach ($qb->getQuery()->getArrayResult() as $item) {
					$orderWarehousePairs[$item['orderId']] = $item['warehouseName'];
				}
			}
		};

		$grid->addColumnText('warehouse', 'eshopProductionWarehouse.ordersGrid.warehouse')
			 ->setRenderer(function(Order $order) use (&$orderWarehousePairs) {
				 return $orderWarehousePairs[$order->getId()] ?? '';
			 })->setFitContent()
			 ->setFilterSelect([null => null] + $this->warehouseRepository->findPairs())
			 ->setCondition(function(QueryBuilder $qb, $value) {
				 $qb->andWhere('wo.warehouse = :wId')
					->setParameter('wId', $value);
			 });
	}

}