<?php declare(strict_types = 1);

namespace EshopProductionWarehouse\AdminModule\Presenters;

use Core\Model\Helpers\Strings;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\AdminModule\Model\Suppliers;
use EshopProductionWarehouse\AdminModule\Components\IWarehouseMovementFormFactory;
use EshopProductionWarehouse\AdminModule\Components\IProductionFormFactory;
use EshopProductionWarehouse\AdminModule\Components\IWarehouseMovementsGridFactory;
use EshopProductionWarehouse\AdminModule\Components\IWarehouseToDateControlFactory;
use EshopProductionWarehouse\AdminModule\Components\IWarehouseTransferFormFactory;
use EshopProductionWarehouse\AdminModule\Components\ProductionForm;
use EshopProductionWarehouse\AdminModule\Components\WarehouseMovementForm;
use EshopProductionWarehouse\AdminModule\Components\WarehouseMovementsGrid;
use EshopProductionWarehouse\AdminModule\Components\WarehouseToDateControl;
use EshopProductionWarehouse\AdminModule\Components\WarehouseTransferForm;
use EshopProductionWarehouse\AdminModule\Model\Common\IOperation;
use EshopProductionWarehouse\AdminModule\Model\Facade\NumericalSeriesFacade;
use EshopProductionWarehouse\AdminModule\Model\Facade\WarehouseFacade;
use EshopProductionWarehouse\AdminModule\Model\FormContainers\ProductsContainer\ProductsContainerConfig;
use EshopProductionWarehouse\Model\Entities\Warehouse;
use EshopProductionWarehouse\Model\Entities\WarehouseMovement;
use EshopProductionWarehouse\Model\Entities\WarehouseMovementInput;
use EshopProductionWarehouse\Model\Entities\WarehouseMovementOutput;
use EshopProductionWarehouse\Model\Utils\Validators;
use Nette\Utils\ArrayHash;

class DefaultPresenter extends BasePresenter
{
	protected ?WarehouseMovement $warehouseMovement = null;

	/** @var NumericalSeriesFacade @inject */
	public NumericalSeriesFacade $numericalSeriesFacade;

	/** @var WarehouseFacade @inject */
	public WarehouseFacade $warehouseFacade;

	/** @var Suppliers @inject */
	public Suppliers $suppliers;

	public function actionDefault(): void
	{
		$this->setHeader('eshopProductionWarehouse.title.warehouse', 'fas fa-warehouse');
		$this['navigation']->setData([
			'header' => [
				[
					'title' => 'eshopProductionWarehouse.menu.addInput',
					'link'  => 'Default:addInputs',
					'ico'   => 'plus'
				],
				[
					'title' => 'eshopProductionWarehouse.menu.addOutput',
					'link'  => 'Default:addOutputs',
					'ico'   => 'minus'
				],
				[
					'title' => 'eshopProductionWarehouse.menu.transfer',
					'link'  => 'Default:transfer',
					'ico'   => 'exchange-alt'
				],
				[
					'title' => 'eshopProductionWarehouse.menu.production',
					'link'  => $this->link('openProductionForm!'),
					'class' => 'ajax',
					'ico'   => 'industry'
				],
				[
					'title' => 'eshopProductionWarehouse.menu.warehouseToDate',
					'link'  => 'Default:warehouseToDate',
					'ico'   => 'calendar-alt'
				],
			]
		]);
	}

	public function actionTransfer(): void
	{
		$this->setHeader('eshopProductionWarehouse.title.transfer', 'fas fa-exchange-alt');
	}

	public function actionAddInputs(): void
	{
		$this->prepareInputsAction();
	}

	public function actionEditInputs($id): void
	{
		$this->prepareInputsAction($id);
	}

	public function actionAddOutputs($id): void
	{
		$this->prepareOutputsAction($id);
	}

	public function actionEditOutputs($id): void
	{
		$this->prepareOutputsAction($id);
	}

	public function actionWarehouseToDate(string $date = null, string $warehouseId = null): void
	{
		$this->setHeader('eshopProductionWarehouse.title.warehouseToDate', 'fas fa-calendar-alt');
	}

	public function handleOpenProductionForm(): void
	{
		$form = $this['productionForm'];
		$form->redrawControl('modal');
	}

	public function createComponentInputsForm(IWarehouseMovementFormFactory $factory): WarehouseMovementForm
	{
		$config = ProductsContainerConfig::create([ProductsContainerConfig::ALLOW_PURCHASE_PRICE => true]);
		$control = $factory->create('eshopProductionWarehouse.warehouseMovementForm.inputs', $config);
		$em = $this->em;

		$control->setOnSetDefaults(function(BaseForm $form, WarehouseMovement $movement) {
			$form['description']->setDefaultValue($movement->inputDescription);
			$form['supplier']->setDefaultValue($movement->inputSupplier ? $movement->inputSupplier->getId() : null);
			$form['invoiceNumber']->setDefaultValue($movement->inputInvoiceNumber);
		});

		$control->setOnValidateCallback(function(BaseForm $form, ArrayHash $values, array $operations, ?WarehouseMovement $movement, WarehouseFacade $warehouseFacade) {
			$newInputIds = [];
			$oldInputIds = $movement ? $movement->getInputIds() : [];

			$products = [];
			/** @var IOperation $operation */
			foreach ($operations as $operation) {
				$defaultInput = $movement ? $movement->getInputById($operation->getId() ?? 0) : null;
				$products[$operation->getProduct()->getId()] = [
					'newQuantity' => $operation->getQuantity(),
					'diffQuantity' => $defaultInput ? $operation->getQuantity() - $defaultInput->getQuantity() : 0
				];
			}

			foreach ($movement ? $movement->getInputs() : [] as $v) { // ty, co byly ve formulari smazany
				$eshopCatalogProductId = $v->getProduct()->getId();
				if (!array_key_exists($eshopCatalogProductId, $products)) {
					$products[$eshopCatalogProductId] = [
						'newQuantity' => 0,
						'diffQuantity' => $v->getQuantity()
					];
				}
			}

			$warehouseState = $warehouseFacade->getState(null, array_keys($products), ($movement ?? $this->em->getRepository(Warehouse::class)->find((int) $values->warehouse))->getId());

			foreach ($warehouseState as $warehouseProductState) {
				$change = $products[$warehouseProductState['id']];
				if (Validators::isDisallowedProductQuantity($warehouseProductState, $change['newQuantity'] - $change['diffQuantity'])) {
					$form->addError($this->t('eshopProductionWarehouse.warehouseMovementForm.errors.negativeProductQuantity', null, ['product' => $warehouseProductState['name']]), false);
				}
			}
		});
		$control->setOnSuccessCallback(function(BaseForm $form, WarehouseFacade $warehouseFacade, WarehouseMovement $movement, array $operations) use ($em) {
			$newInputIds = [];
			$oldInputIds = $movement->getInputIds();

			$movement->inputInvoiceNumber = $form->values->invoiceNumber;
			$movement->inputDescription = $form->values->description;
			$movement->inputSupplier = $this->suppliers->get((int) $form->values->supplier);

			/** @var IOperation[] $operations */
			foreach ($operations as $product) {
				$eshopCatalogProduct = $product->getProduct();
				$eshopCatalogProductId = $eshopCatalogProduct->getId();
				$diffQuantity = 0;
				if ($product->getId()) {
					/** @var WarehouseMovementInput|null $input */
					$input = $em->getRepository(WarehouseMovementInput::class)->find($product->getId());
					$newInputIds[] = $product->getId();
					$diffQuantity = (int) ($product->getQuantity() - $input->getQuantity());
				} else {
					$input = new WarehouseMovementInput($product->getProduct(), $movement, $product->getQuantity());
					$movement->addInput($input);
					$diffQuantity = (int) $input->getQuantity();
				}

				$input->price = Strings::formatEntityDecimal($product->getExtras()['price']);
				$input->setProduct($product->getProduct());
				$input->setQuantity($product->getQuantity());

				$eshopCatalogProduct->quantity += $diffQuantity;
				$em->persist($eshopCatalogProduct);
			}

			foreach (array_diff($oldInputIds, $newInputIds) as $inputId) {
				$input = $em->getRepository(WarehouseMovementInput::class)->find((int) $inputId);
				if (!$input) {
					continue;
				}
				$movement->removeInput($input);
			}
		});

		if ($this->warehouseMovement) {
			$operations = $this->warehouseMovement->getInputs();
			foreach ($operations as $operation) {
				$operation->setExtra('price', (float) $operation->price);
				$operation->setExtra('averagePrice', $this->warehouseFacade->getAveragePurchasePrice($operation->getProduct()->getId()));
			}
			$control->setWarehouseMovement($this->warehouseMovement, $operations);
		}

		$control['form']->onSuccessSave[] = function(BaseForm $form) {
			$this->redirect('this');
			$this->presenter->redrawControl('flashes');
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->redirect('default');
			$this->presenter->redrawControl('flashes');
		};
		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}

	public function createComponentOutputsForm(IWarehouseMovementFormFactory $factory): WarehouseMovementForm
	{
		$config = ProductsContainerConfig::create();
		$control = $factory->create('eshopProductionWarehouse.warehouseMovementForm.outputs', $config);
		$em = $this->em;

		$control->setOnSetDefaults(function(BaseForm $form, WarehouseMovement $movement) {
			$form['description']->setDefaultValue($movement->outputDescription);
			$form['supplier']->setDefaultValue($movement->outputSupplier ? $movement->outputSupplier->getId() : null);
			$form['invoiceNumber']->setDefaultValue($movement->outputInvoiceNumber);
		});

		$control->setOnValidateCallback(function(BaseForm $form, ArrayHash $values, array $operations, ?WarehouseMovement $movement, WarehouseFacade $warehouseFacade) {
			$newOutputIds = [];
			$oldOutputIds = $movement ? $movement->getOutputIds() : [];

			$products = [];
			/** @var IOperation $operation */
			foreach ($operations as $operation) {
				$defaultInput = $movement ? $movement->getOutputById($operation->getId() ?? 0) : null;
				$products[$operation->getProduct()->getId()] = [
					'newQuantity' => $operation->getQuantity(),
					'diffQuantity' => -($defaultInput ? $operation->getQuantity() - $defaultInput->getQuantity() : $operation->getQuantity())
				];
			}

			foreach ($movement ? $movement->getInputs() : [] as $v) { // ty, co byly ve formulari smazany
				$eshopCatalogProductId = $v->getProduct()->getId();
				if (!array_key_exists($eshopCatalogProductId, $products)) {
					$products[$eshopCatalogProductId] = [
						'newQuantity' => 0,
						'diffQuantity' => -$v->getQuantity()
					];
				}
			}

			$warehouseState = $warehouseFacade->getState(null, array_keys($products), ($movement ?? $this->em->getRepository(Warehouse::class)->find((int) $values->warehouse))->getId());

			foreach ($warehouseState as $warehouseProductState) {
				$change = $products[$warehouseProductState['id']];
				if (Validators::isDisallowedProductQuantity($warehouseProductState, $change['diffQuantity'])) {
					$form->addError($this->t('eshopProductionWarehouse.warehouseMovementForm.errors.negativeProductQuantity', null, ['product' => $warehouseProductState['name']]), false);
				}
			}
		});
		$control->setOnSuccessCallback(function(BaseForm $form, WarehouseFacade $warehouseFacade, WarehouseMovement $movement, array $operations) use ($em) {
			$newOutputIds = [];
			$oldOutputIds = $movement->getOutputIds();

			$movement->outputInvoiceNumber = $form->values->invoiceNumber;
			$movement->outputDescription = $form->values->description;
			$movement->outputSupplier = $this->suppliers->get((int) $form->values->supplier);

			/** @var IOperation[] $operations */
			foreach ($operations as $product) {
				$eshopCatalogProduct = $product->getProduct();
				$eshopCatalogProductId = $eshopCatalogProduct->getId();
				$diffQuantity = 0;
				if ($product->getId()) {
					/** @var WarehouseMovementOutput|null $output */
					$output = $em->getRepository(WarehouseMovementOutput::class)->find($product->getId());
					$newOutputIds[] = $product->getId();
					$diffQuantity = (int) ($product->getQuantity() - $output->getQuantity());
				} else {
					$output = new WarehouseMovementOutput($product->getProduct(), $movement, $product->getQuantity());
					$movement->addOutput($output);
					$diffQuantity = (int) $output->getQuantity();
				}

				$output->setProduct($product->getProduct());
				$output->setQuantity($product->getQuantity());

				$eshopCatalogProduct->quantity -= $diffQuantity;
				$em->persist($eshopCatalogProduct);
			}

			foreach (array_diff($oldOutputIds, $newOutputIds) as $outputId) {
				$output = $em->getRepository(WarehouseMovementOutput::class)->find((int) $outputId);
				if (!$output) {
					continue;
				}
				$movement->removeOutput($output);
			}
		});

		if ($this->warehouseMovement) {
			$control->setWarehouseMovement($this->warehouseMovement, $this->warehouseMovement->getOutputs());
		}

		$control['form']->onSuccessSave[] = function(BaseForm $form) {
			$this->redirect('this');
			$this->presenter->redrawControl('flashes');
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->redirect('default');
			$this->presenter->redrawControl('flashes');
		};
		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}

	public function createComponentWarehouseMovementsGrid(IWarehouseMovementsGridFactory $factory): WarehouseMovementsGrid
	{
		return $factory->create();
	}

	public function createComponentProductionForm(IProductionFormFactory $factory): ProductionForm
	{
		$control = $factory->create();
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->getPresenter()['warehouseMovementsGrid']['grid']->reload();
			$this->getPresenter()->payload->hideModal = true;
			$this->getPresenter()->redrawControl('flashes');
		};
		return $control;
	}

	public function createComponentWarehouseToDateControl(IWarehouseToDateControlFactory $factory): WarehouseToDateControl
	{
		return $factory->create();
	}

	public function createComponentWarehouseTransferForm(IWarehouseTransferFormFactory $factory): WarehouseTransferForm
	{
		$control = $factory->create();
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->redirect('default');
			$this->presenter->redrawControl('flashes');
		};
		return $control;
	}

	protected function prepareInputsAction($id = null): void
	{
		if ($id) {
			$this->warehouseMovement = $this->em->getRepository(WarehouseMovement::class)->find((int) $id);
			$header = $this->t('eshopProductionWarehouse.title.editInput', null, ['ident' => $this->warehouseMovement->inputNumber ?? $this->warehouseMovement->date->format('d.m.Y')]);
		} else {
			$header = $this->t('eshopProductionWarehouse.title.addInput', null, ['ident' => $this->numericalSeriesFacade->getInputNumber()]);
		}
		$this->setHeader($header, 'fas fa-truck-loading');
	}

	protected function prepareOutputsAction($id = null): void
	{
		if ($id) {
			$this->warehouseMovement = $this->em->getRepository(WarehouseMovement::class)->find((int) $id);
			$header = $this->t('eshopProductionWarehouse.title.editOutput', null, ['ident' => $this->warehouseMovement->outputNumber ?? $this->warehouseMovement->date->format('d.m.Y')]);
		} else {
			$header = $this->t('eshopProductionWarehouse.title.addOutput', null, ['ident' => $this->numericalSeriesFacade->getOutputNumber()]);
		}
		$this->setHeader($header, 'fas fa-truck-loading');
	}

}
