<?php declare(strict_types = 1);

namespace EshopProductionWarehouse\Model\Subscribers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\Event;
use EshopCatalog\Model\Entities\Product;
use EshopOrders\Model\Entities\Order;
use EshopProductionWarehouse\AdminModule\Model\Facade\WarehouseFacade;
use EshopProductionWarehouse\Model\Entities\WarehouseMovement;
use EshopProductionWarehouse\Model\Entities\WarehouseMovementInput;
use EshopProductionWarehouse\Model\Entities\WarehouseMovementOutput;
use Nette\Utils\DateTime;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class PseudoWarehouseSubscriber implements EventSubscriberInterface
{
	protected EntityManagerDecorator $em;
	protected WarehouseFacade $warehouseFacade;

	public function __construct(EntityManagerDecorator $em, WarehouseFacade $warehouseFacade)
	{
		$this->em = $em;
		$this->warehouseFacade = $warehouseFacade;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'pseudoWarehouse.afterReduceProductsQuantity'   => ['afterReduceProductsQuantity', 100],
			'pseudoWarehouse.afterIncreaseProductsQuantity' => ['afterIncreaseProductsQuantity', 100],
		];
	}

	public function afterReduceProductsQuantity(Event $event): void
	{
		/** @var Order|null $order */
		$order = $event->data['order'];

		$warehouse = $order ? $this->warehouseFacade->getWarehouseByOrder($order->getId()) : null;
		
		/** @var array<int, array{product: Product, quantityDiff: int}> $changes */
		$changes = $event->data['changes'];

		$warehouseMovement = new WarehouseMovement(new DateTime);
		$warehouseMovement->warehouse = $warehouse;

		foreach ($changes as $change) {
			$warehouseMovementOutput = new WarehouseMovementOutput($change['product'], $warehouseMovement, $change['quantityDiff']);
			$warehouseMovement->addOutput($warehouseMovementOutput);
		}

		if (count($changes) > 0) {
			$this->em->persist($warehouseMovement)->flush($warehouseMovement);
		}
	}

	public function afterIncreaseProductsQuantity(Event $event): void
	{
		/** @var Order|null $order */
		$order = $event->data['order'];

		$warehouse = $order ? $this->warehouseFacade->getWarehouseByOrder($order->getId()) : null;

		/** @var array<int, array{product: Product, quantityDiff: int}> $changes */
		$changes = $event->data['changes'];

		$warehouseMovement = new WarehouseMovement(new DateTime);
		$warehouseMovement->warehouse = $warehouse;

		foreach ($changes as $change) {
			$warehouseMovementInput = new WarehouseMovementInput($change['product'], $warehouseMovement, $change['quantityDiff']);
			$warehouseMovement->addInput($warehouseMovementInput);
		}

		if (count($changes) > 0) {
			$this->em->persist($warehouseMovement)->flush($warehouseMovement);
		}
	}

}