<?php declare(strict_types = 1);

namespace EshopProductsComparison\FrontModule\Model\Export;

use Contributte\Translation\Translator;
use Core\Model\Dao\SiteDomain;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Sites;
use Currency\Model\Config;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Parameter;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\FrontModule\Model\ProductQuery;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\Model\Entities\Category;
use EshopCatalog\Model\Entities\Product;
use EshopOrders\Model\PaymentSpeditions;
use EshopProductsComparison\Model\Entities\CategoryExport;
use EshopProductsComparison\Model\Entities\ManufacturerExport;
use EshopProductsComparison\Model\Entities\ProductExport;
use EshopProductsComparison\Model\EshopProductsComparisonConfig;
use Nette\Utils\Json;

class Data
{
	protected ?array $cCategories                    = null;
	public ?string   $currency                       = null;
	protected ?array $cDisabledManufacturers         = null;
	protected ?array $cCategoryExports               = null;
	protected ?array $cCategoryExportsLocale         = null;
	protected array  $cServiceCategoriesByNames      = [];
	protected array  $cServiceCategoriesByIds        = [];
	protected array  $cServiceCategoriesByNamesShort = [];

	public function __construct(
		protected EntityManagerDecorator $em,
		protected Translator             $translator,
		protected Sites                  $sites,
		protected Categories             $categories,
		protected ProductsFacade         $productsFacade,
		protected PaymentSpeditions      $paymentSpeditions,
	)
	{
	}

	public function getProductsData(int $start = 0, int $limit = 1000): array
	{
		$allIds                   = [];
		$exports                  = [];
		$result                   = [
			'products' => [],
		];
		$minimumPrice             = EshopProductsComparisonConfig::load('exportMinimumPrice', 1);
		$defaultDomainCurrency    = null;
		$useCategoryExportsLocale = false;

		$currentSite = $this->sites->getCurrentSite();
		if ($currentSite->getCurrentDomain() instanceof SiteDomain) {
			$defaultDomainCurrency = $currentSite->getCurrentDomain()->defaultCurrency;
		}

		foreach ((new ProductQuery($this->translator->getLocale()))
			         ->onlyInStockOrSupplier()
			         ->selectIds()
			         ->inSite($this->sites->getCurrentSite()->getIdent())
			         ->hasPrice()
			         ->getQueryBuilder($this->em->getRepository(Product::class))
			         ->setMaxResults($limit)->getQuery()->setFirstResult($start)->getScalarResult() as $row) {
			/** @var array $row */
			$allIds[] = $row['id'];
		}

		$result['baseCount'] = count($allIds);

		if ($this->cDisabledManufacturers === null) {
			$this->cDisabledManufacturers = [];

			foreach ($this->em->getRepository(ManufacturerExport::class)->createQueryBuilder('me')
				         ->select('IDENTITY(me.id) as id, me.service, me.allowExport, me.lang')
				         ->getQuery()->getArrayResult() as $row) {
				/** @var array $row */
				if ($row['allowExport'] === 0) {
					$this->cDisabledManufacturers[$row['id']][$row['service']] = true;
				}
			}
		}

		if ($this->cCategoryExports === null) {
			$this->cCategoryExports = [];

			foreach ($this->em->getRepository(CategoryExport::class)->createQueryBuilder('ce')
				         ->where('ce.lang = :lang')
				         ->setParameters(new ArrayCollection([new Parameter('lang', $this->translator->getLocale())]))
				         ->getQuery()->getArrayResult() as $row) {
				/** @var array $row */
				$this->cCategoryExports[$row['id']][$row['service']] = $row;
			}
		}

		if (Categories::$overrideLocale && $this->translator->getLocale() !== Categories::$overrideLocale) {
			$useCategoryExportsLocale = true;

			if ($this->cCategoryExportsLocale === null) {
				$this->cCategoryExportsLocale = [];

				foreach ($this->em->getRepository(CategoryExport::class)->createQueryBuilder('ce')
					         ->where('ce.lang = :lang')
					         ->setParameters(new ArrayCollection([new Parameter('lang', Categories::$overrideLocale)]))
					         ->getQuery()->getArrayResult() as $row) {
					/** @var array $row */
					$this->cCategoryExportsLocale[$row['id']][$row['service']] = $row;
				}
			}

			if (empty($this->cServiceCategoriesByIds)) {
				$this->loadServiceCategories('heureka', Categories::$overrideLocale);
				$this->loadServiceCategories('heureka', $this->translator->getLocale());
			}
		}

		if ($allIds) {
			foreach (array_chunk($allIds, 800) as $ids) {

				foreach ($this->em->getRepository(ProductExport::class)->createQueryBuilder('pe')
					         ->where('pe.id IN (' . implode(',', $ids) . ')')->andWhere('pe.lang = :lang')
					         ->setParameters(new ArrayCollection([new Parameter('lang', $this->translator->getLocale())]))
					         ->getQuery()->getArrayResult() as $row) {
					/** @var array $row */
					$exports[$row['id']][$row['service']] = $row;
				}

				foreach ($this->productsFacade->getProducts($ids) as $product) {
					if (
						!$product->isActive
						|| !$product->link
						|| $product->link === '#'
						|| $product->priceInBaseCurrency < $minimumPrice
					) {
						continue;
					}

					$urlJoiner = \str_contains($product->link, '?') ? '&' : '?';
					if (Config::load('disableCurrParameter')) {
						if ($this->currency && $this->currency !== $defaultDomainCurrency) {
							$product->link .= $urlJoiner . 'curr=' . $this->currency;
						}
					} else if ($this->currency) {
						$product->link .= $urlJoiner . 'curr=' . $this->currency;
					}

					$category = $product->defaultCategory;
					if (!$category) {
						continue;
					}

					$catExports = $this->cCategoryExports[$product->defaultCategoryId] ?? [];
					$needLoop   = [];
					$cat        = $category;
					while (count($needLoop) !== 4 && $cat) {
						if (isset($this->cCategoryExports[$cat->id])) {
							foreach ($this->cCategoryExports[$cat->id] as $service => $data) {
								if ($useCategoryExportsLocale && !$data['categoryText'] && isset($this->cCategoryExportsLocale[$cat->id][$service]['categoryText']) && Categories::$overrideLocale) {
									$dataLocale = $this->cCategoryExportsLocale[$cat->id][$service];

									$serviceCatId = $this->getServiceCategoryByName($service, Categories::$overrideLocale, $dataLocale['categoryText']);
									if (!$serviceCatId) {
										$tmp          = explode(' | ', $dataLocale['categoryText']);
										$serviceCatId = $this->getServiceCategoryByNameShort($service, Categories::$overrideLocale, end($tmp));
									}

									if ($serviceCatId) {
										$serviceCatText = $this->getServiceCategoryById($service, $this->translator->getLocale(), $serviceCatId);

										if ($serviceCatText) {
											$data                 = $dataLocale;
											$data['categoryText'] = $serviceCatText;
										}
									}
								}

								if (!isset($needLoop[$service])) {
									$catExports[$service] = $data;
								}
								if ($data['status'] !== 2) {
									$needLoop[$service] = false;
								}
							}
						}

						$cat = $cat->getParent();
					}

					if (Categories::$overrideLocale && isset($catExports['heureka']) && $catExports['heureka']['categoryText'] && !str_starts_with($catExports['heureka']['categoryText'], 'Heureka')) {
						$tmp          = explode(' | ', $catExports['heureka']['categoryText']);
						$serviceCatId = $this->getServiceCategoryByNameShort('heureka', Categories::$overrideLocale, end($tmp));

						if ($serviceCatId) {
							$serviceCatText = $this->getServiceCategoryById('heureka', $this->translator->getLocale(), $serviceCatId);

							if ($serviceCatText) {
								$catExports['heureka']['categoryText'] = $serviceCatText;
							}
						}
					}

					$product->addExtraField('export', $exports[$product->getId()] ?? []);
					$product->addExtraField('category', $category);
					$product->addExtraField('categoryExports', $catExports);
					$product->addExtraField('disabledByManufacturer', $this->cDisabledManufacturers[$product->manufacturerId] ?? []);

					$result['products'][] = $product;
				}
			}

			$this->em->clear();
			gc_collect_cycles();
		}

		return $result;
	}

	public function getCategories(): array
	{
		if ($this->cCategories === null) {
			$this->cCategories = [];

			$exports = [];
			foreach ($this->em->getRepository(CategoryExport::class)->createQueryBuilder('ce')
				         ->where('ce.lang = :lang')->setParameter('lang', $this->translator->getLocale())
				         ->getQuery()->getArrayResult() as $row) {
				/** @var array $row */
				$exports[$row['id']][$row['service']] = $row;
			}

			$categories = [];
			foreach ($this->em->getRepository(Category::class)->createQueryBuilder('c')
				         ->select('c.id, IDENTITY(c.parent) as parent, c.isPublished, ct.name')
				         ->innerJoin('c.categoryTexts', 'ct', 'WITH', 'ct.lang = :lang')
				         ->setParameter('lang', $this->translator->getLocale())
				         ->getQuery()->getArrayResult() as $row) {
				/** @var array $row */
				$categories[$row['id']] = $row;
			}

			foreach ($categories as $category) {
				$name      = $category['name'];
				$export    = [];
				$parent    = $categories[$category['parent']];
				$parentIds = [];

				if (isset($exports[$category['id']])) {
					$export[] = $exports[$category['id']];
				}

				$i = 0;
				while ($parent && $i < 20) {
					$parentIds[] = $parent['id'];
					$name        = $parent['name'] . ' | ' . $name;

					$parent = $categories[$parent['parent']];
					$i++;
				}

				$data = [
					'categoryText' => $name,
					'status'       => $category['isPublished'] ? 1 : 0,
					'exports'      => [],
				];

				foreach ($parentIds as $id) {
					if (isset($exports[$id])) {
						$export[] = $exports[$id];
					}
				}

				foreach ($export as $y => $vals) {
					foreach ($vals as $k => $v) {
						$data['exports'][$y][$k] = [
							'categoryText' => $v['categoryText'] ?: $data['categoryText'],
							'status'       => $v['status'],
						];
					}
				}

				$this->cCategories[$category['id']] = $data;
			}
		}

		return $this->cCategories;
	}

	protected function getServiceCategoryByName(string $service, string $lang, string $name): ?int
	{
		$key = $service . '_' . $lang;

		return $this->cServiceCategoriesByNames[$key][$name] ?? null;
	}

	protected function getServiceCategoryByNameShort(string $service, string $lang, string $name): ?int
	{
		$key = $service . '_' . $lang;

		return $this->cServiceCategoriesByNamesShort[$key][$name] ?? null;
	}

	protected function getServiceCategoryById(string $service, string $lang, int $id): ?string
	{
		$key = $service . '_' . $lang;

		return $this->cServiceCategoriesByIds[$key][$id] ?? null;
	}

	protected function loadServiceCategories(string $service, string $lang): void
	{
		$key = $service . '_' . $lang;

		if (!array_key_exists($key, $this->cServiceCategoriesByIds)) {
			$this->cServiceCategoriesByIds[$key]        = [];
			$this->cServiceCategoriesByNames[$key]      = [];
			$this->cServiceCategoriesByNamesShort[$key] = [];

			try {
				$data = file_get_contents('https://api-evidence.pshk.cz/v1/cms/' . $service . '-export-categories/' . $lang);
				if ($data) {
					/** @var array $json */
					$json = Json::decode((string) $data, forceArrays: true) ?? [];

					foreach ($json['data'] ?? [] as $k => $v) {
						/** @var string|int $k */
						/** @var string $v */

						if ($service === 'heureka' && !str_starts_with($v, 'Heureka')) {
							continue;
						}

						$this->cServiceCategoriesByIds[$key][$k]   = $v;
						$this->cServiceCategoriesByNames[$key][$v] = (int) $k;

						$tmp                                                         = explode(' | ', $v);
						$this->cServiceCategoriesByNamesShort[$key][trim(end($tmp))] = (int) $k;
					}
				}
			} catch (\Exception $e) {
			}
		}
	}
}
