<?php declare(strict_types = 1);

namespace EshopProductsComparison\FrontModule\Model\Export;

use Currency\Model\Entities\Currency;
use DOMDocument;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\FrontModule\Model\Dao\Tag;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Helpers\ProductConditionEnum;
use EshopProductsComparison\FrontModule\Model\Event\XmlShopItemEvent;
use EshopProductsComparison\Model\EshopProductsComparisonConfig;
use Gallery\FrontModule\Model\Dao\Album;
use Nette\Utils\DateTime;
use Nette\Utils\Strings;
use Override;

class GoogleXmlService extends XmlService
{
	protected string $name = 'google';

	#[Override]
	public function startExport(): void
	{
		parent::startExport();

		$siteName = $this->domain->siteName;

		$this->writeToFile('<?xml version="1.0" encoding="utf-8"?>' . PHP_EOL .
			'<rss version="2.0" xmlns:g="http://base.google.com/ns/1.0">' . PHP_EOL .
			'<channel>' . PHP_EOL .
			'<title>' . $siteName . '</title>' . PHP_EOL .
			'<link>' . $this->baseUrl . '</link>' . PHP_EOL .
			'<description>Produkty obchodu ' . $siteName . '</description>' . PHP_EOL);
	}

	#[Override]
	public function writeItem(Product $product): void
	{
		if (isset($this->usedProducts[$product->getId()])) {
			return;
		}

		$this->usedProducts[$product->getId()] = true;

		$export = $this->prepareExportData($product);

		if (
			$export['status'] === 0
			|| array_key_exists($this->name, (array) $product->getExtraField('disabledByManufacturer') ?: [])
		) {
			return;
		}

		$currencyEntity = $this->getCurrencyEntity();
		if (!$currencyEntity instanceof Currency) {
			return;
		}

		$dom               = new DOMDocument();
		$dom->formatOutput = true;
		$dom->encoding     = 'utf-8';
		$shopitem          = $dom->createElement('item');

		$child = $dom->createElement('g:id', (string) $product->getId());
		$shopitem->appendChild($child);

		$child     = $dom->createElement('g:title');
		$childText = $dom->createTextNode((string) $export['product']);
		$child->appendChild($childText);
		$shopitem->appendChild($child);

		$child     = $dom->createElement('g:description');
		$childText = $dom->createCDATASection(Strings::normalize(strip_tags(html_entity_decode((string) ($export['productDescription'] ?: $product->getDescription())))));
		$child->appendChild($childText);
		$shopitem->appendChild($child);

		$child     = $dom->createElement('g:link');
		$childText = $dom->createCDATASection((string) $product->link);
		$child->appendChild($childText);
		$shopitem->appendChild($child);

		$canUseRetailPrice = !Config::load('enablePriceHistory') || !Config::load('product.allowLoadRetailPrice');

		if ($canUseRetailPrice && $product->getRetailPrice() > $product->getPrice()) {
			$child = $dom->createElement('g:price', number_format(round($product->getRetailPrice(), $currencyEntity->decimals), 2, ".", "") . ' ' . $this->currency);
			$shopitem->appendChild($child);
			$child = $dom->createElement('g:sale_price', number_format(round($product->getPrice(), $currencyEntity->decimals), 2, ".", "") . ' ' . $this->currency);
			$shopitem->appendChild($child);
		} else {
			$child = $dom->createElement('g:price', number_format(round($product->getPrice(), $currencyEntity->decimals), 2, ".", "") . ' ' . $this->currency);
			$shopitem->appendChild($child);
		}

		foreach ($this->getSpeditionsForPrice($product->getPrice(), $product->isOversize, $product->hasDisabledPickupPoints(), $product->disabledSpeditions, $product->disabledPayments) as $country => $speditions) {
			$country = Strings::upper($country);
			if ($country !== Strings::upper($this->domain->getDefaultCountry())) {
				continue;
			}

			foreach ($speditions as $spedName => $spedition) {
				$delivery = $dom->createElement('g:shipping');

				$child = $dom->createElement('g:country', $country);
				$delivery->appendChild($child);
				$child = $dom->createElement('g:service', (string) $spedName);
				$delivery->appendChild($child);
				$child = $dom->createElement('g:price', number_format(round($spedition['price'], $currencyEntity->decimals), 2, '.', '') . ' ' . $this->currency);
				$delivery->appendChild($child);

				$shopitem->appendChild($delivery);

				$configShipping = EshopProductsComparisonConfig::loadArray('google.shippingData.' . $country) ?: [];
				foreach ($configShipping as $key => $value) {
					$child = $dom->createElement('g:' . $key, (string) $value);
					$delivery->appendChild($child);
				}
			}
		}

		$gAv = $product->getAvailability()?->getOpenGraphText();
		if (!$gAv) {
			if ($product->getAvailability()?->getIdent() === 'inStock') {
				$gAv = 'inStock';
			} else if ($product->getAvailability()?->getDelay() > 0) {
				$gAv = 'backorder';
			} else {
				$gAv = 'outOfStock';
			}
		}

		if (in_array($gAv, ['inStock', 'in stock'])) {
			$gAv = 'in_stock';
			//			if ($product->getQuantity() <= 2) {
			//				$gAv = 'limited_availability';
			//			}
		} else if (in_array($gAv, ['outOfStock', 'out of stock', 'soldOut', 'sold out', 'sold_out'])) {
			$gAv = 'out_of_stock';
		} else if (in_array($gAv, ['preOrder', 'pre order'])) {
			$gAv = 'preorder';
		} else if (in_array($gAv, ['backOrder', 'back order'])) {
			$gAv = 'backorder';
		}

		$child = $dom->createElement('g:availability', $gAv);
		$shopitem->appendChild($child);

		if (in_array($gAv, ['preorder', 'backorder']) && $product->getAvailability()?->getDelay()) {
			$date = (new DateTime)->modify('+' . ($product->getAvailability()->getDelay()) . ' days');

			$child = $dom->createElement('g:availability_date', $date->format('Y-m-d'));
			$shopitem->appendChild($child);
		}

		$child = $dom->createElement('g:condition', ProductConditionEnum::toGoogle($product->condition));
		$shopitem->appendChild($child);

		if ($product->getGallery() instanceof Album) {
			$coverId = null;

			if ($product->getGallery()->getCover() && $product->getGallery()->getCover()->getFilePath()) {
				$coverId = (int) $product->getGallery()->getCover()->id;
				$child   = $dom->createElement('g:image_link', $this->parseImgUrl($product->getGallery()->getCover()->getFilePath()));
				$shopitem->appendChild($child);
			}

			foreach ($product->getGallery()->getImages() as $img) {
				if ((int) $img->id === $coverId || !$img->getFilePath()) {
					continue;
				}

				$child = $dom->createElement('g:additional_image_link', $this->parseImgUrl($img->getFilePath()));
				$shopitem->appendChild($child);
			}
		}

		if ($product->getEan()) {
			$child = $dom->createElement('g:gtin', $product->getEan());
			$shopitem->appendChild($child);
		}

		if ($export['categoryText']) {
			$child = $dom->createElement('g:google_product_category', (string) $export['categoryText']);
			$shopitem->appendChild($child);
		}

		$cats = [];
		$cat  = $product->defaultCategory;

		while ($cat) {
			$cats[] = $cat->name;
			$cat    = $cat->getParent();
		}

		if ($cats !== []) {
			$cats = array_reverse($cats);

			$child = $dom->createElement('g:product_type', implode(' > ', $cats));
			$shopitem->appendChild($child);
		}

		if ($product->getManufacturer() && trim($product->getManufacturer()->name)) {
			$child = $dom->createElement('g:brand', $product->getManufacturer()->name);
			$shopitem->appendChild($child);
		}

		$usedFeatures = [];
		$colors       = [];
		foreach ($product->getFeatures() as $feature) {
			if (
				!$this->canExportFeature((int) $feature->idFeature)
				|| isset($usedFeatures[$feature->idFeature])
				|| !$feature->showInExport
			) {
				continue;
			}

			$featureValue = $this->getExportFeatureValue((int) $feature->idFeatureValue) ?: $feature->value;
			$googleType   = $this->getGoogleFeatureType((int) $feature->idFeature);

			if ($googleType) {
				$value = null;
				if ($googleType === 'color') {
					$colors[] = $featureValue;
					continue;
				}
				if (in_array($googleType, ['material', 'pattern', 'size', 'size_type'])) {
					$value = $featureValue;
				} else if (in_array($googleType, [
					'gender',
					'age_group',
					'is_bundle',
					'energy_efficiency_class',
					'min_energy_efficiency_class',
					'min_energy_efficiency_class',
					'size_system',
				])) {
					$value = $this->getGoogleFeatureValueType((int) $feature->idFeatureValue);
				}
				if ($value) {
					$param = $dom->createElement('g:' . $googleType, $value);
					$shopitem->appendChild($param);
				}
			} else if ($featureValue) {
				$param = $dom->createElement('g:product_detail');
				$child = $dom->createElement('g:attribute_name', (string) ($this->getExportFeatureName((int) $feature->idFeature) ?: $feature->name));
				$param->appendChild($child);
				$child = $dom->createElement('g:attribute_value', htmlspecialchars((string) $featureValue, ENT_XML1 | ENT_QUOTES, 'UTF-8'));
				$param->appendChild($child);
				$shopitem->appendChild($param);
				$usedFeatures[$feature->idFeature] = $feature->idFeatureValue;
			}
		}

		if ($colors !== []) {
			$param = $dom->createElement('g:color', implode('/', $colors));
			$shopitem->appendChild($param);
		}


		if ($product->getTag('freeDelivery') instanceof Tag) {
			$param = $dom->createElement('g:shipping_label', $this->translator->translate('eshopProductsComparison.freeDelivery'));
			$shopitem->appendChild($param);
		}

		if ($product->variantId) {
			$param = $dom->createElement('g:item_group_id', $product->variantId);
			$shopitem->appendChild($param);
		}

		foreach ([0, 1, 2, 3, 4] as $cfI) {
			if ($export['customLabel' . $cfI]) {
				$param = $dom->createElement('g:custom_label_' . $cfI, $export['customLabel' . $cfI]);
				$shopitem->appendChild($param);
			}
		}

		$event = new XmlShopItemEvent($dom, $shopitem, $product, 'google');
		$this->eventDispatcher->dispatch($event, 'eshopProductsComparison.xmlShopItem');

		$dom->appendChild($shopitem);

		$this->writeToFile($dom->saveXML($dom->documentElement) . PHP_EOL);
	}

	#[Override]
	public function endExport(): void
	{
		$this->writeToFile('</channel>' . PHP_EOL . '</rss>');

		parent::endExport();
	}
}
