<?php declare(strict_types = 1);

namespace EshopProductsComparison\FrontModule\Model;

use Core\Model\Dao\Site;
use Core\Model\Dao\SiteDomain;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\Event;
use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\BaseService;
use Core\Model\Notifiers\MailNotifiers\LogNotifier;
use Currency\Model\Currencies;
use Currency\Model\Entities\Currency;
use EshopProductsComparison\FrontModule\Model\Export\Data;
use EshopProductsComparison\FrontModule\Model\Export\XmlExportCollection;
use EshopProductsComparison\FrontModule\Model\Export\XmlService;
use EshopProductsComparison\Model\EshopProductsComparisonConfig;
use Exception;
use Nette\Http\Request;
use Tracy\Debugger;

class ExportFacade extends BaseService
{
	public ?string $currency     = null;
	public array   $writtenLinks = [];

	public function __construct(
		protected EntityManagerDecorator $em,
		protected XmlExportCollection    $xmlExport,
		protected Request                $httpRequest,
		protected Data                   $dataService,
		protected EventDispatcher        $eventDispatcher,
		protected Currencies             $currencies,
	)
	{
	}

	public function generateExport(array $types, Site $site, SiteDomain $domain): void
	{
		if (php_sapi_name() !== 'cli') {
			set_time_limit(6000);
		}

		if ($this->currency) {
			$this->dataService->currency = $this->currency;
		}

		$currencyEntity = $this->currencies->getCurrent();
		if (!$currencyEntity instanceof Currency) {
			throw new Exception('Currency not found');
		}

		// Načtení vybraných exportů
		/** @var XmlService[] $services */
		$services = [];
		/** @var XmlService $v */
		foreach ($this->xmlExport->getServices() as $k => $v) {
			if (in_array($k, $types)) {
				$services[$k] = $v;
			}
		}

		// Inicializace exportu
		$baseUrl = 'https://' . $domain->getDomain();

		foreach ($services as $service) {
			$service->setSite($site);
			$service->setDomain($domain);
			$service->setBaseUrl($baseUrl);
			$service->setCurrency($this->currency);
			$service->setCurrencyEntity($currencyEntity);
			$service->startExport();
		}

		$i          = 1;
		$start      = 0;
		$limit      = EshopProductsComparisonConfig::loadInt('productsLoopCount') ?: 5000;
		$loop       = 1;
		$loopLimit  = 100;
		$empty      = 0;
		$emptyLimit = EshopProductsComparisonConfig::loadInt('emptyLimit') ?: 6;

		$products = $this->dataService->getProductsData($start, $limit);

		do {
			$nextLoop = $products['baseCount'] >= $limit && $loop <= $loopLimit;

			foreach ($products['products'] as $product) {
				if (!isset($this->writtenLinks[$product->link])) {
					foreach ($services as $service) {
						$service->writeItem($product);
					}
				}

				$this->writtenLinks[$product->link][] = $product->getId();
			}

			$start    = $limit * $i++;
			$products = $this->dataService->getProductsData($start, $limit);

			$loop++;

			$this->em->clear();
			gc_collect_cycles();
		} while ($nextLoop);

		foreach ($services as $service) {
			$service->endExport();
		}

		$event = new Event();
		$this->eventDispatcher->dispatch($event, 'eshopProductsComparison.exportFinished');
	}
}
