<?php declare(strict_types = 1);

namespace EshopProductsComparison\AdminModule\Components\AdminTools;

use Core\Model\Lang\Langs;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductTexts;
use EshopProductsComparison\Model\Entities\ProductExport;
use EshopProductsComparison\Model\Helpers\ExportEnums;
use Exception;
use Nette\Http\FileUpload;
use Nette\Utils\ArrayHash;
use Nette\Utils\Html;

class HeurekaProductDataImportForm extends BaseControl
{
	protected Langs $langs;

	public function __construct(
		Langs $langs
	)
	{
		$this->langs = $langs;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();
		$form->setShowLangSwitcher(false);

		$form->addUpload('file', 'eshopProductsComparison.adminTools.seoHeureka.file');
		$form->addText('lines', 'eshopProductsComparison.adminTools.seoHeureka.lines')
			->setHtmlType('number')
			->setDefaultValue(30)
			->setRequired();
		$form->addSelect('lang', 'eshopProductsComparison.adminTools.seoHeureka.lang', $this->langs->getOptionsForSelect());

		$form->addSubmit('upload', 'default.upload');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		$result  = [];
		$okCount = 0;
		try {
			$data = [];
			/** @var FileUpload $file */
			$file = $values->file;

			if ($file->hasFile() && $file->isOk()) {
				$fr        = fopen($file->getTemporaryFile(), 'r');
				$i         = 0;
				$separator = ';';
				while (!feof($fr) && $line = fgetcsv($fr, null, $separator)) {
					if ($i === 0) {
						if (count($line) === 1) {
							$separator = ',';
						}
						$i++;
						continue;
					}

					if (empty($line[10])) {
						continue;
					}

					$data[(int) $line[6]] = [
						'name'     => (string) $line[11],
						'category' => (string) $line[12],
					];
					if ($i === (int) $values->lines) {
						break;
					}
					$i++;
				}
				fclose($fr);
			}

			if ($data) {
				/** @var ProductExport[] $exports */
				$exports = [];

				/** @var Product[] $allProducts */
				$allProducts = [];

				/** @var ProductTexts[] $texts */
				$texts = [];

				foreach ($this->em->getRepository(Product::class)->findBy([
					'id'        => array_keys($data),
					'isDeleted' => 0,
				]) as $row) {
					$allProducts[$row->getId()] = $row;
				}

				foreach ($this->em->getRepository(ProductTexts::class)->findBy([
					'id'   => array_keys($data),
					'lang' => $values->lang,
				]) as $row) {
					$texts[$row->getProduct()->getId()] = $row;
				}

				foreach ($this->em->getRepository(ProductExport::class)->createQueryBuilder('ppe')
					         ->where('ppe.id IN (:ids)')
					         ->andWhere('ppe.lang = :lang')
					         ->andWhere('ppe.service = :service')
					         ->setParameters([
						         'ids'     => array_keys($data),
						         'lang'    => $values->lang,
						         'service' => 'heureka',
					         ])
					         ->getQuery()->getResult() as $row) {
					/** @var ProductExport $row */
					$exports[$row->id->getId()] = $row;
				}

				foreach ($data as $id => $vals) {
					if (!isset($allProducts[$id])) {
						$result[] = [
							'status'      => 'ERROR',
							'statusClass' => 'danger',
							'message'     => "Produkt $id nenalezen",
						];
						continue;
					}

					if (isset($exports[$id])) {
						$export   = $exports[$id];
						$result[] = [
							'status'      => 'OK',
							'statusClass' => 'success',
							'message'     => Html::el()
								->setHtml("Data produktu <a href='{$this->presenter->link(':EshopCatalog:Admin:Products:edit', ['id' => $id])}' target='_blank'>{$texts[$id]->name}</a> aktualizována"),
						];
					} else {
						$export   = new ProductExport($allProducts[$id], $values->lang, 'heureka');
						$result[] = [
							'status'      => 'OK',
							'statusClass' => 'success',
							'message'     => Html::el()
								->setHtml("Data produktu <a href='{$this->presenter->link(':EshopCatalog:Admin:Products:edit', ['id' => $id])}' target='_blank'>{$texts[$id]->name}</a> vytvořena"),
						];
					}

					if ($vals['name']) {
						$export->product     = $vals['name'];
						$export->productName = $vals['name'];
					}

					if ($vals['category']) {
						$export->categoryText = $vals['category'];
					}

					$okCount++;
					$export->status = ExportEnums::STATUS_ACTIVE;
					$this->em->persist($export);
				}

				$this->em->flush();
			} else {
				$result[] = [
					'status'      => 'ERROR',
					'statusClass' => 'danger',
					'message'     => 'Soubor nelze načíst',
				];
			}

			$result[] = [
				'status'      => 'INFO',
				'statusClass' => 'info',
				'message'     => "Upraveno $okCount/{$values->lines}",
			];

			$this->template->result = $result;
			$this->redrawControl('form');

			return true;
		} catch (Exception $e) {
			$this->presenter->flashMessageSuccess($e->getMessage());
			$this->presenter->redrawControl('flashes');
			$this->redrawControl('form');
		}

		return false;
	}
}
