<?php declare(strict_types = 1);

namespace EshopProductsComparison\AdminModule\Model\Subscribers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\SetFormDataEvent;
use Core\Model\UI\Form\BaseContainer;
use EshopCatalog\AdminModule\Components\Categories\CategoryForm;
use EshopCatalog\Model\Entities\Category;
use EshopProductsComparison\Model\Entities\CategoryExport;
use EshopProductsComparison\Model\Helpers\ExportEnums;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class CategoryFormSubscriber implements EventSubscriberInterface
{
	protected EntityManagerDecorator $em;

	public function __construct(EntityManagerDecorator $em)
	{
		$this->em = $em;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			CategoryForm::class . '::createForm'  => 'createForm',
			CategoryForm::class . '::formSuccess' => 'formSuccess',
			CategoryForm::class . '::setCategory' => 'setCategory',
		];
	}

	public function createForm(CreateFormEvent $event): void
	{
		$baseContainer = $event->form->addContainer('categoryComparison');
		$baseContainer->addCustomData('template', __DIR__ . '/CategoryForm.latte');

		foreach (ExportEnums::$services as $serviceName => $v) {
			$container = new BaseContainer;
			$container->addBool('status', 'eshopProductsComparison.comparisons.status', [
				0 => ['title' => 'default.no', 'btnClass' => 'danger'],
				1 => ['title' => 'default.yes', 'btnClass' => 'success'],
				2 => ['title' => 'eshopProductsComparison.categoryForm.fromParent', 'btnClass' => 'secondary'],
			])
				->setDefaultValue(2)
				->setIsMultilanguage();
			$container->addText('categoryText', 'eshopProductsComparison.comparisons.' . $serviceName . '.categoryText')
				->setHtmlAttribute('data-autocomplete-url', 'https://api-evidence.pshk.cz/v1/cms/' . $serviceName . '-export-categories/[data-content-lang]')
				->setMaxLength(255)
				->setIsMultilanguage();

			if ($serviceName === ExportEnums::SERVICE_FACEBOOK) {
				$container->getComponent('categoryText')
					->setHtmlAttribute('data-select-item-data', 'id,name')
					->setIsMultilanguage(false);
			}


			if ($serviceName === ExportEnums::SERVICE_GOOGLE) {
				foreach ([0, 1, 2, 3, 4] as $cfI) {
					$container->addText(
						'customLabel' . $cfI,
						'eshopProductsComparison.comparisons.' . $serviceName . '.customLabel' . $cfI
					)
						->setMaxLength(255)
						->setIsMultilanguage();
				}
			} else if ($serviceName === ExportEnums::SERVICE_FACEBOOK) {
				foreach ([0, 1, 2, 3, 4] as $cfI) {
					$container->addText(
						'customLabel' . $cfI,
						'eshopProductsComparison.comparisons.' . $serviceName . '.customLabel' . $cfI
					)
						->setMaxLength(255)
						->setIsMultilanguage();
				}
			} else if ($serviceName === ExportEnums::SERVICE_ZBOZI) {
				foreach ([1] as $cfI) {
					$container->addText(
						'customLabel' . $cfI,
						'eshopProductsComparison.comparisons.' . $serviceName . '.customLabel' . $cfI
					)
						->setMaxLength(255)
						->setIsMultilanguage();
				}
			}

			$baseContainer->addComponent($container, $serviceName);
		}
	}

	public function formSuccess(FormSuccessEvent $event): void
	{
		/** @var Category $category */
		$category   = $event->custom['entity'];
		$values     = $event->values;
		$langValues = [];
		foreach ($event->form->convertMultilangValuesToArray() as $l => $vals) {
			$langValues[$l] = $vals['categoryComparison'];
		}

		$fbCategoryText = $event->form->getValues()->categoryComparison->facebook->categoryText ?? null;

		$exports    = [];
		$exportVals = [];
		foreach ($this->em->getRepository(CategoryExport::class)->createQueryBuilder('cpe')
			         ->where('cpe.id = :id')->setParameter('id', $category->getId())
			         ->getQuery()->getResult() as $row) {
			$exports[$row->service][$row->lang] = $row;
		}

		foreach (ExportEnums::$services as $serviceName => $v) {
			foreach ($langValues as $lang => $vals) {
				if (isset($vals[$serviceName])) {
					$exportVals[$lang][$serviceName] = $vals[$serviceName];
				}

				if ($serviceName === ExportEnums::SERVICE_FACEBOOK && $fbCategoryText) {
					$exportVals[$lang][$serviceName]['categoryText'] = $fbCategoryText;
				}
			}
		}

		foreach ($exportVals as $lang => $services) {
			foreach ($services as $serviceName => $vals) {
				if (isset($exports[$serviceName][$lang])) {
					$export = $exports[$serviceName][$lang];
				} else {
					$export = new CategoryExport($category, $lang, $serviceName);
				}

				$export->status       = (int) ($vals['status'] ?? 2);
				$export->categoryText = $vals['categoryText'] ?: null;

				foreach ([0, 1, 2, 3, 4] as $cfI) {
					$export->{'customLabel' . $cfI} = $vals['customLabel' . $cfI] ?: null;
				}

				$this->em->persist($export);
			}
		}
	}

	public function setCategory(SetFormDataEvent $event): void
	{
		/** @var Category $category */
		$category = $event->entity;
		$form     = $event->form->getComponent('categoryComparison');

		$exports = [];
		foreach ($this->em->getRepository(CategoryExport::class)->createQueryBuilder('cpe')
			         ->where('cpe.id = :id')->setParameter('id', $category->getId())
			         ->getQuery()->getArrayResult() as $row) {
			foreach ($row as $serviceName => $v) {
				if (in_array($serviceName, ['service', 'lang', 'id'])) {
					continue;
				}

				$exports[$row['service']][$serviceName][$row['lang']] = $v;
			}
		}

		$form->setDefaults($exports);
	}
}
