<?php declare(strict_types = 1);

namespace EshopProductsComparison\AdminModule\Model\Subscribers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\SetFormDataEvent;
use Core\Model\Lang\Langs;
use Core\Model\UI\Form\BaseContainer;
use EshopCatalog\AdminModule\Components\Manufacturers\ManufacturerForm;
use EshopCatalog\Model\Entities\Manufacturer;
use EshopProductsComparison\Model\Entities\ManufacturerExport;
use EshopProductsComparison\Model\Helpers\ExportEnums;
use Nette\Localization\Translator;
use Nette\Utils\ArrayHash;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class ManufacturerFormSubscriber implements EventSubscriberInterface
{
	protected EntityManagerDecorator $em;
	protected Langs                  $langs;
	protected Translator             $translator;

	public function __construct(
		EntityManagerDecorator $em,
		Langs                  $langs,
		Translator             $translator
	)
	{
		$this->em         = $em;
		$this->langs      = $langs;
		$this->translator = $translator;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			ManufacturerForm::class . '::createForm'      => 'createForm',
			ManufacturerForm::class . '::formSuccess'     => 'formSuccess',
			ManufacturerForm::class . '::setManufacturer' => 'setManufacturer',
		];
	}

	public function createForm(CreateFormEvent $event): void
	{
		$baseContainer = $event->form->addContainer('manufacturerComparison');

		foreach (ExportEnums::$services as $serviceName => $v) {
			$container = new BaseContainer;
			$container->addBool('allowExport', $this->translator->translate('eshopProductsComparison.comparisons.status') . ' - ' . $this->translator->translate($v['title']), [
				0 => ['title' => $this->translator->translate('default.no'), 'btnClass' => 'danger'],
				1 => ['title' => $this->translator->translate('default.yes'), 'btnClass' => 'success'],
			])
				->setDefaultValue(1)
				->setTranslator(null);

			$baseContainer->addComponent($container, $serviceName);
		}
	}

	public function formSuccess(FormSuccessEvent $event): void
	{
		/** @var Manufacturer $manufacturer */
		$manufacturer = $event->custom['entity'];

		/** @var ArrayHash $values */
		$values = $event->values;

		$exports    = [];
		$exportVals = [];
		foreach ($this->em->getRepository(ManufacturerExport::class)->createQueryBuilder('me')
			         ->where('me.id = :id')->setParameter('id', $manufacturer->getId())
			         ->getQuery()->getResult() as $row) {
			$exports[$row->service][$row->lang] = $row;
		}

		foreach (ExportEnums::$services as $serviceName => $v) {
			foreach ($this->langs->getLangs(false) as $lang) {
				if (isset($values->manufacturerComparison->{$serviceName})) {
					$exportVals[$lang->getTag()][$serviceName] = $values->manufacturerComparison->{$serviceName};
				}
			}
		}

		foreach ($exportVals as $lang => $services) {
			foreach ($services as $serviceName => $vals) {
				if (isset($exports[$serviceName][$lang])) {
					$export = $exports[$serviceName][$lang];
				} else {
					$export = new ManufacturerExport($manufacturer, $lang, $serviceName);
				}

				$export->allowExport = (int) ($vals['allowExport'] ?? 1);

				$this->em->persist($export);
			}
		}
	}

	public function setManufacturer(SetFormDataEvent $event): void
	{
		/** @var Manufacturer $manufacturer */
		$manufacturer = $event->entity;
		$form         = $event->form->getComponent('manufacturerComparison');

		$exports = [];
		foreach ($this->em->getRepository(ManufacturerExport::class)->createQueryBuilder('me')
			         ->where('me.id = :id')->setParameter('id', $manufacturer->getId())
			         ->getQuery()->getArrayResult() as $row) {
			$exports[$row['service']]['allowExport'] = $row['allowExport'];
		}

		$form->setDefaults($exports);
	}
}
