<?php declare(strict_types = 1);

namespace EshopProductsComparison\Console;

use Core\Model\Lang\Langs;
use Core\Model\Sites;
use Currency\Model\Currencies;
use Doctrine;
use Doctrine\ORM\Tools\Console\Command\SchemaTool\AbstractCommand;
use EshopCatalog\FrontModule\Model\Categories;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopProductsComparison\FrontModule\Model\ExportFacade;
use EshopProductsComparison\Model\EshopProductsComparisonConfig;
use Nette\Localization\Translator;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

class GenerateExport extends AbstractCommand
{
	/** @var Sites|null @inject */
	public ?Sites $sites = null;

	/** @var ExportFacade|null @inject */
	public ?ExportFacade $exportFacade = null;

	/** @var Translator|null @inject */
	public ?Translator $translator = null;

	/** @var Langs|null @inject */
	public ?Langs $langs = null;

	/**
	 * @return void
	 */
	protected function configure()
	{
		$this->setName('eshopproductscomparison:generate-export')
			->setDescription('')
			->addOption('types', 'y', InputOption::VALUE_REQUIRED)
			->addOption('site', 's', InputOption::VALUE_REQUIRED)
			->addOption('lang', 'l', InputOption::VALUE_REQUIRED)
			->addOption('currency', 'c', InputOption::VALUE_REQUIRED)
			->addOption('linkLang', 'll', InputOption::VALUE_OPTIONAL);
	}

	/**
	 * @return int
	 */
	protected function executeSchemaCommand(InputInterface $input, OutputInterface $output, Doctrine\ORM\Tools\SchemaTool $schemaTool, array $metadatas, SymfonyStyle $ui)
	{
		ini_set('memory_limit', EshopProductsComparisonConfig::load('consoleMemoryLimit', '2G'));
		$output->writeln('GENERATE EXPORT');

		try {
			$siteIdent = $input->getOption('site');
			$lang      = $input->getOption('lang');
			$currency  = $input->getOption('currency');
			$types     = explode(',', $input->getOption('types'));
			$linkLang  = $input->getOption('linkLang') ?: null;

			Sites::$currentIdentOverride     = $siteIdent;
			Sites::$currentLangOverride      = $lang;
			Currencies::$currentCodeOverride = $currency;
			$this->exportFacade->currency    = $currency;
			/** @phpstan-ignore-next-line */
			$this->translator->setLocale($lang);
			$this->langs->setDefault($lang);

			if ($linkLang) {
				ProductsFacade::$forceLinkLocale = $linkLang;
				Categories::$overrideLocale      = $linkLang;
			}

			$site = $this->sites->getSites()[$siteIdent] ?? null;
			if (!$site) {
				return Command::FAILURE;
			}

			$domain = $site->getDomains()[$lang] ?? null;
			if (!$domain) {
				$domain = $site->getCurrentDomain();
			}

			$domain->setLang($lang);

			$this->exportFacade->generateExport($types, $site, $domain);

			$output->writeLn('<info>[OK] - generated</info>');

			return Command::SUCCESS;
		} catch (\Exception $e) {
			$output->writeln('<error>[ERROR] - ' . $e->getMessage() . '</error>');
		}

		return Command::FAILURE;
	}
}
