<?php declare(strict_types = 1);

namespace EshopProductsComparison\FrontModule\Model\Export;

use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\Model\Entities\Availability;
use EshopProductsComparison\FrontModule\Model\Event\XmlShopItemEvent;
use EshopSales\FrontModule\Model\Dao\OrderSale;
use EshopSales\FrontModule\Model\OrderSales;
use Nette\Utils\DateTime;
use Nette\Utils\Strings;

class HeurekaXmlService extends XmlService
{
	public string $name = 'heureka';

	protected ?array $specialServices = null;

	public function startExport(): void
	{
		parent::startExport();

		$this->writeToFile('<?xml version="1.0" encoding="utf-8"?>' . PHP_EOL .
			'<SHOP>' . PHP_EOL);

		$this->startExportStock();
	}

	public function startExportStock(): void
	{
		parent::startExportStock();

		$this->writeToFileStock('<?xml version="1.0" encoding="utf-8"?>' . PHP_EOL .
			'<item_list>' . PHP_EOL);
	}

	public function writeItem(Product $product)
	{
		if (isset($this->usedProducts[$product->getId()])) {
			return;
		}

		$currencyEntity = $this->getCurrencyEntity();
		if (!$currencyEntity) {
			return;
		}

		$this->usedProducts[$product->getId()] = true;

		$export   = $this->prepareExportData($product);
		$quantity = $product->getQuantity() + $product->getQuantityExternal();

		if (
			$export['status'] === 0
			|| array_key_exists($this->name, $product->getExtraField('disabledByManufacturer') ?? [])
		) {
			return;
		}

		$dom               = new \DOMDocument();
		$dom->formatOutput = true;
		$dom->encoding     = 'utf-8';
		$shopitem          = $dom->createElement('SHOPITEM');

		$child = $dom->createElement('ITEM_ID', (string) $product->getId());
		$shopitem->appendChild($child);

		if ($product->variantId) {
			$child = $dom->createElement('ITEMGROUP_ID', \Core\Model\Helpers\Strings::truncate(Strings::webalize((string) $product->variantId, null, false), 36, ''));
			$shopitem->appendChild($child);
		}

		$child     = $dom->createElement('PRODUCTNAME');
		$childText = $dom->createTextNode((string) $export['productName']);
		$child->appendChild($childText);
		$shopitem->appendChild($child);

		$child     = $dom->createElement('PRODUCT');
		$childText = $dom->createTextNode((string) $export['product']);
		$child->appendChild($childText);
		$shopitem->appendChild($child);

		$child     = $dom->createElement('DESCRIPTION');
		$childText = $dom->createCDATASection(Strings::normalize(strip_tags(html_entity_decode((string) ($export['productDescription'] ?: $product->getDescription())))));
		$child->appendChild($childText);
		$shopitem->appendChild($child);

		$child     = $dom->createElement('URL');
		$childText = $dom->createCDATASection((string) $product->link);
		$child->appendChild($childText);
		$shopitem->appendChild($child);

		$price = $product->getPrice();

		if (
			!empty($product->tags)
			&& class_exists('\EshopSales\FrontModule\Model\OrderSales')
			&& $this->container->hasService('eshopSales.front.orderSales')
		) {
			/** @var OrderSales $salesService */
			$salesService = $this->container->getService('eshopSales.front.orderSales');
			$salesService->loadProductTagSalePrice($product);

			if ($product->moreData['tagSalesPrice']) {
				/** @var OrderSale $dto */
				$dto       = $product->moreData['tagSalesPrice']['dto'];
				$codeValue = $dto->code ?: $dto->getAutoSaleCode();

				if ($codeValue) {
					$salesVoucher = $dom->createElement('SALES_VOUCHER');

					$price = $product->moreData['tagSalesPrice']['price'];

					$code = $dom->createElement('CODE', $codeValue);
					$salesVoucher->appendChild($code);

					if ($dto->description) {
						$desc = $dom->createElement('DESC', Strings::truncate($dto->description, 125));
						$salesVoucher->appendChild($desc);
					}

					$shopitem->appendChild($salesVoucher);
				}
			}
		}

		$child = $dom->createElement('PRICE_VAT', number_format(round($price, $currencyEntity->decimals), 2, ".", ""));
		$shopitem->appendChild($child);

		if ($product->getVatRate()) {
			$child = $dom->createElement('VAT', $product->getVatRate() . '%');
			$shopitem->appendChild($child);
		}

		$child = $dom->createElement('DELIVERY_DATE', (string) $product->getAvailability()->getDelay());
		$shopitem->appendChild($child);

		foreach ($this->getSpeditionsForPrice($product->getPrice(), $product->isOversize, $product->hasDisabledPickupPoints(), $product->disabledSpeditions, $product->disabledPayments) as $country => $speditions) {
			if (Strings::upper($country) !== Strings::upper($this->domain->getDefaultCountry())) {
				continue;
			}

			foreach ($speditions as $spedName => $spedition) {
				$delivery = $dom->createElement('DELIVERY');

				$child = $dom->createElement('DELIVERY_ID', (string) $spedName);
				$delivery->appendChild($child);
				$child = $dom->createElement('DELIVERY_PRICE', number_format(round($spedition['price'], $currencyEntity->decimals), 2, '.', ''));
				$delivery->appendChild($child);

				if ($spedition['cod']) {
					$child = $dom->createElement('DELIVERY_PRICE_COD', (string) $spedition['cod']);
					$delivery->appendChild($child);
				}

				$shopitem->appendChild($delivery);
			}
		}

		if ($product->getGallery()) {
			$coverId = null;
			if ($product->getGallery()->getCover()) {
				$coverId = $product->getGallery()->getCover()->id;
				$child   = $dom->createElement('IMGURL', $this->parseImgUrl($product->getGallery()->getCover()->getFilePath()));
				$shopitem->appendChild($child);
			}

			foreach ($product->getGallery()->getImages() as $img) {
				if ($img->id === $coverId) {
					continue;
				}

				$child = $dom->createElement('IMGURL_ALTERNATIVE', $this->parseImgUrl($img->getFilePath()));;
				$shopitem->appendChild($child);
			}
		}

		if ($product->getEan()) {
			$child = $dom->createElement('EAN', (string) $product->getEan());
			$shopitem->appendChild($child);
		}

		if ($export['categoryText']) {
			$child = $dom->createElement('CATEGORYTEXT', (string) $export['categoryText']);
			$shopitem->appendChild($child);
		}

		if ($product->getManufacturer() && trim($product->getManufacturer()->name)) {
			$child = $dom->createElement('MANUFACTURER', (string) $product->getManufacturer()->name);
			$shopitem->appendChild($child);
		}

		$usedFeatures = [];
		foreach ($product->getFeatures() as $feature) {
			if (
				!$this->canExportFeature((int) $feature->idFeature)
				|| isset($usedFeatures[$feature->idFeature])
				|| !$feature->showInExport
			) {
				continue;
			}

			$featureValue = $this->getExportFeatureValue((int) $feature->idFeatureValue) ?: $feature->value;

			$param = $dom->createElement('PARAM');

			$child = $dom->createElement('PARAM_NAME', $this->getExportFeatureName((int) $feature->idFeature) ?: $feature->name);
			$param->appendChild($child);
			$child = $dom->createElement('VAL', htmlspecialchars($featureValue, ENT_XML1 | ENT_QUOTES, 'UTF-8'));
			$param->appendChild($child);

			$shopitem->appendChild($param);
			$usedFeatures[$feature->idFeature] = $feature->idFeatureValue;
		}

		if ($export['bidCpc']) {
			$child = $dom->createElement('HEUREKA_CPC', (string) $export['bidCpc']);
			$shopitem->appendChild($child);
		}

		foreach ($product->getGifts() as $gift) {
			/** @phpstan-ignore-next-line */
			$child = $dom->createElement('GIFT', Strings::truncate($gift->getName(), 250, ''));
			/** @phpstan-ignore-next-line */
			$child->setAttribute('ID', (string) $gift->getProduct()->getId());
			$shopitem->appendChild($child);
		}

		foreach ($this->canAddSpecialServices() as $text) {
			$child = $dom->createElement('SPECIAL_SERVICE', $text);
			$shopitem->appendChild($child);
		}

		$event = new XmlShopItemEvent($dom, $shopitem, $product, 'heureka');
		$this->eventDispatcher->dispatch($event, 'eshopProductsComparison.xmlShopItem');

		$dom->appendChild($shopitem);

		$this->writeToFile($dom->saveXML($dom->documentElement) . PHP_EOL);

		// Stock
		if ($quantity <= 0 && $product->canAddToCart && $product->getAvailability()->getIdent() === Availability::IN_STOCK) {
			$quantity = 5;
		}

		if ($quantity > 0) {
			$dom               = new \DOMDocument();
			$dom->formatOutput = true;
			$dom->encoding     = 'utf-8';

			$shopitem = $dom->createElement('item');
			$shopitem->setAttribute('id', (string) $product->getId());

			$child = $dom->createElement('stock_quantity', (string) $quantity);
			$shopitem->appendChild($child);

			$dom->appendChild($shopitem);

			$this->writeToFileStock($dom->saveXML($dom->documentElement) . PHP_EOL);
		}
	}

	public function endExport()
	{
		$this->writeToFile('</SHOP>');

		parent::endExport();

		$this->endExportStock();
	}

	public function endExportStock()
	{
		$this->writeToFileStock('</item_list>');

		parent::endExportStock();
	}

	protected function canAddSpecialServices(): array
	{
		if ($this->specialServices === null) {
			$this->specialServices = [];

			$zboziSettingsKey        = 'eshopCatalog' . Strings::firstUpper($this->site->getIdent()) . 'XmlFeedHeureka';
			$zboziSpecialServicesKey = $zboziSettingsKey . 'SpecialService';

			$now = (new DateTime())
				->setTime(0, 0, 0)
				->format('Y-m-d H:i');

			for ($i = 1; $i <= 5; $i++) {
				$specialService       = $this->settings->get($zboziSpecialServicesKey . $i);
				$specialServiceDateTo = $this->settings->get($zboziSpecialServicesKey . 'DateTo' . $i);

				$toDate = $specialServiceDateTo ? DateTime::createFromFormat('d.m.Y H:i', $specialServiceDateTo) : null;

				if (is_array($specialService)) {
					$specialServiceText = $specialService[$this->translator->getLocale()] ?? null;

					if ($specialServiceText && (!$toDate || $now <= $toDate->format('Y-m-d H:i'))) {
						$this->specialServices[] = Strings::truncate($specialServiceText, 128, '');
					}
				}
			}
		}

		return $this->specialServices;
	}
}
