<?php declare(strict_types = 1);

namespace EshopProductsComparison\FrontModule\Model\Export;

use Core\Model\Helpers\Arrays;
use Currency\Model\Currencies;
use EshopCatalog\FrontModule\Model\Dao\Product;
use EshopCatalog\Model\Entities\Availability;
use EshopCatalog\Model\Helpers\ProductConditionEnum;
use EshopOrders\Model\Entities\OrderDiscount;
use EshopProductsComparison\FrontModule\Model\Event\XmlShopItemEvent;
use EshopSales\FrontModule\Model\Dao\OrderSale;
use EshopSales\FrontModule\Model\OrderSales;
use Nette\Utils\Strings;

class IdealoXmlService extends XmlService
{
	public string $name = 'idealo';

	public function startExport(): void
	{
		parent::startExport();

		$this->writeToFile('<?xml version="1.0" encoding="utf-8"?>' . PHP_EOL .
			'<SHOP>' . PHP_EOL);
	}

	public function writeItem(Product $product)
	{
		if (isset($this->usedProducts[$product->getId()])) {
			return;
		}

		$this->usedProducts[$product->getId()] = true;

		$export   = $this->prepareExportData($product);
		$quantity = $product->getQuantity() + $product->getQuantityExternal();

		if (
			$export['status'] === 0
			|| array_key_exists($this->name, $product->getExtraField('disabledByManufacturer') ?? [])
		) {
			return;
		}

		$dom               = new \DOMDocument();
		$dom->formatOutput = true;
		$dom->encoding     = 'utf-8';
		$shopitem          = $dom->createElement('SHOPITEM');

		$child = $dom->createElement('sku', (string) $product->getId());
		$shopitem->appendChild($child);

		if ($product->getManufacturer() && trim($product->getManufacturer()->name)) {
			$child = $dom->createElement('brand', (string) $product->getManufacturer()->name);
			$shopitem->appendChild($child);
		}

		$child     = $dom->createElement('title');
		$childText = $dom->createTextNode((string) $export['product']);
		$child->appendChild($childText);
		$shopitem->appendChild($child);

		if ($export['categoryText']) {
			$child     = $dom->createElement('categoryPath');
			$childText = $dom->createCDATASection((string) $export['categoryText']);
			$child->appendChild($childText);
			$shopitem->appendChild($child);
		}

		$child     = $dom->createElement('url');
		$childText = $dom->createCDATASection((string) $product->link);
		$child->appendChild($childText);
		$shopitem->appendChild($child);

		if ($product->getEan()) {
			$child = $dom->createElement('eans', (string) $product->getEan());
			$shopitem->appendChild($child);

			$child = $dom->createElement('gtin', (string) $product->getEan());
			$shopitem->appendChild($child);
		}

		$child     = $dom->createElement('description');
		$childText = $dom->createCDATASection(Strings::normalize(strip_tags(html_entity_decode((string) $product->shortDescription))));
		$child->appendChild($childText);
		$shopitem->appendChild($child);

		$price = $product->getPrice();

		if (
			!empty($product->tags)
			&& class_exists('\EshopSales\FrontModule\Model\OrderSales')
			&& $this->container->hasService('eshopSales.front.orderSales')
		) {
			/** @var OrderSales $salesService */
			$salesService = $this->container->getService('eshopSales.front.orderSales');

			/** @var Currencies $currencies */
			$currencies = $this->container->getService('currency.currencies');

			$salesService->loadProductTagSalePrice($product);

			if ($product->moreData['tagSalesPrice']) {
				/** @var OrderSale $dto */
				$dto = $product->moreData['tagSalesPrice']['dto'];

				$sale = '';
				if ($dto->type === OrderDiscount::TYPE_PERCENT) {
					$sale = $dto->value . ' %';
				} else if ($dto->type === OrderDiscount::TYPE_FIX) {
					$sale = $dto->value . ' ' . $currencies->getCurrent()->symbol;
				}
				$codeValue = $dto->code ?: $dto->getAutoSaleCode();

				if ($dto->description) {
					$child = $dom->createElement('voucherCode', Strings::truncate($this->translator->translate('eshopProductsComparisonFront.exportSaleText', [
						'sale' => $sale,
						'code' => $codeValue,
					]), 128));
					$shopitem->appendChild($child);
				}
			}
		}

		$child = $dom->createElement('price', number_format($price, 2, ".", ""));
		$shopitem->appendChild($child);

		$child = $dom->createElement('deliveryTime', '1-3 Werktage');
		$shopitem->appendChild($child);

		if ($product->getGallery()) {
			$coverId = null;
			if ($product->getGallery()->getCover()) {
				$coverId = $product->getGallery()->getCover()->id;
				$child   = $dom->createElement('imageUrl', $this->parseImgUrl($product->getGallery()->getCover()->getFilePath()));
				$shopitem->appendChild($child);
			}

			foreach ($product->getGallery()->getImages() as $img) {
				if ($img->id === $coverId) {
					continue;
				}

				$child = $dom->createElement('imageUrl', $this->parseImgUrl($img->getFilePath()));
				$shopitem->appendChild($child);
			}
		}

		$child = $dom->createElement('conditionType', ProductConditionEnum::toIdealo($product->condition));
		$shopitem->appendChild($child);

		if ($quantity <= 0 && $product->canAddToCart && $product->getAvailability()->getIdent() === Availability::IN_STOCK) {
			$quantity = 5;
		}

		$child = $dom->createElement('quantity', (string) $quantity);
		$shopitem->appendChild($child);

		foreach ($this->getSpeditionsForPrice($product->getPrice(), $product->isOversize, $product->hasDisabledPickupPoints(), $product->disabledSpeditions, $product->disabledPayments) as $country => $speditions) {
			if (Strings::upper($country) !== Strings::upper($this->domain->getDefaultCountry())) {
				continue;
			}

			foreach ($speditions as $spedName => $spedition) {
				$k     = $spedName === 'empty' ? 'deliveryCosts' : 'deliveryCosts_' . $spedName;
				$child = $dom->createElement($k, (string) $spedition['price']);
				$shopitem->appendChild($child);
			}
		}

		$usedFeatures = [];
		$colors       = [];
		foreach ($product->getFeatures() as $feature) {
			if (
				!$this->canExportFeature((int) $feature->idFeature)
				|| isset($usedFeatures[$feature->idFeature])
				|| !$feature->showInExport
			) {
				continue;
			}

			$featureValue = $this->getExportFeatureValue((int) $feature->idFeatureValue) ?: $feature->value;
			$googleType   = $this->getGoogleFeatureType((int) $feature->idFeature);

			if ($googleType && Arrays::contains(['color', 'gender', 'material'], $googleType)) {
				if ($googleType === 'color') {
					$colors[] = $featureValue;
				} else {
					$child = $dom->createElement($googleType, $featureValue);
					$shopitem->appendChild($child);
				}
			} else if ($featureValue) {
				$param = $dom->createElement('param' . $feature->idFeature);

				$child = $dom->createElement('name', $this->getExportFeatureName((int) $feature->idFeature) ?: $feature->name);
				$param->appendChild($child);
				$child = $dom->createElement('value', $featureValue);
				$param->appendChild($child);

				$shopitem->appendChild($param);
				$usedFeatures[$feature->idFeature] = $feature->idFeatureValue;
			}
		}

		if (!empty($colors)) {
			$param = $dom->createElement('colour', implode('/', $colors));
			$shopitem->appendChild($param);
		}

		$event = new XmlShopItemEvent($dom, $shopitem, $product, 'idealo');
		$this->eventDispatcher->dispatch($event, 'eshopProductsComparison.xmlShopItem');

		$dom->appendChild($shopitem);

		$this->writeToFile($dom->saveXML($dom->documentElement) . PHP_EOL);
	}

	public function endExport()
	{
		$this->writeToFile('</SHOP>');

		parent::endExport();
	}
}
